import React from 'react';
import ReactDOM from 'react-dom';

const app = require('../../../../app');
const OwnerSuggestView = require('../../filter/suggest/OwnerSuggestView');
const AuthorSuggestView = require('../../filter/suggest/AuthorSuggestView');
const BasicFilterLayout = require('../../filter/BasicFilterLayout');
const FormAcceptBehavior = require('../../../behaviors/forms/FormAcceptBehavior');
const TaskTypeAutocompleteView = require('../../filter/selects/autocomplete/TaskTypeAutocompleteView');

import { SuggestTagMulti } from '../../../../components/SuggestTag';
import i18n from '../../../../components/i18n';

/**
 * @class SchedulersListFilterView
 * @extends BasicFilterLayout
 */
const SchedulersListFilterView = BasicFilterLayout.extend({

    tagName: 'form',
    className: 'schedulers_list_filter',

    template: require('./tpl/SchedulersListFilterView.hbs'),

    ui: _.extend({}, BasicFilterLayout.prototype.ui, {
        schedulerTags: '.scheduler_tags',
        schedulerId: '.scheduler_id',
        schedulerIdInput: '.scheduler_id input',
        status: '.scheduler_status'
    }),

    events: _.extend({}, BasicFilterLayout.prototype.events, {
        'bemmy:refresh @ui.schedulerId': 'onIdChange',
        'bemmy:set @ui.status': 'onStatusChange'
    }),

    modelEvents: {
        change: 'onModelChange',
        blank: 'render',
        'change:tags': 'renderControls',
        'change:all_tags': 'renderControls'
    },

    behaviors: {
        FormAcceptBehavior: {}
    },

    initialize() {
        this.listenTo(this, FormAcceptBehavior.EVENTS.ACCEPT_EVENT, this.onFormAccept);
        this.renderControls = this.renderControls.bind(this);
        this.handleTagsSet = this.handleTagsSet.bind(this);
        this.handleTagsAllSet = this.handleTagsAllSet.bind(this);
    },

    onRender() {
        this.$('.input').bemmyInput();
        this.$('.select').bemmySelect();

        this.$el.validate({
            onChange: true,
            eachValidField() {
                $(this).removeClass('not-valid');
            },
            eachInvalidField() {
                $(this).addClass('not-valid');
            }
        });

        this.ownerSuggest.close();
        this.ownerSuggest.show(new OwnerSuggestView({
            model: this.model,
            ownerGroups: this.options.props.groups.convertToList(),
            defaultValue: this.model.get('owner'),
            modelTargetKey: 'owner'
        }));

        this.authorSuggest.close();
        this.authorSuggest.show(new AuthorSuggestView({
            model: this.model,
            defaultValue: this.model.get('author'),
            modelTargetKey: 'author'
        }));

        this.typeSelector.close();
        this.typeSelector.show(new TaskTypeAutocompleteView({
            lazy: 400,
            items: this.options.props.types,
            value: this.model.get('task_type'),
            model: this.model,
            placeHolder: 'Task type',
            showAnyItem: true,
            modelTargetKey: 'task_type',
            selectClassName: 'app_filter_type'
        }));

        this.mapModelToView();

        this.renderControls();
    },

    renderControls(model, tags) {
        const _tags = this.model.get('tags');
        const togglerValue = Boolean(this.model.get('all_tags'));

        if (this.ui.schedulerTags[0]) {
            ReactDOM.render(
                <SuggestTagMulti
                    value={tags || (_tags instanceof Array ? _tags : _tags.split(','))}
                    togglerValue={togglerValue}
                    name={'tags'}
                    placeholder={i18n.labels.tags}
                    onChange={this.handleTagsSet}
                    onToggle={this.handleTagsAllSet}
                    />,
                this.ui.schedulerTags[0]
            );
        }
    },

    handleTagsSet(tags) {
        this.model.set('tags', tags);
    },

    handleTagsAllSet(flag) {
        this.model.set('all_tags', flag);
    },

    onIdChange() {
        this.model.set('id', this.ui.schedulerIdInput.val());
        this.ui.schedulerIdInput.change();
    },

    onStatusChange(evtObj, value) {
        this.model.set('status', value);
    },

    mapModelToView() {
        this.ui.schedulerIdInput.val(this.model.get('id'));
        this.ui.status.trigger('bemmy:set', this.model.get('status'));
    },

    serializeData() {
        return _.extend({}, this.model.toJSON(), {
            statuses: this.options.props.statuses
        });
    },

    /**
     * @override
     * @param {Backbone.Model} filterValues - filter model
     */
    updateLastFilter(filterValues) {
        const defaultFilter = app.request('DEFAULT_FILTER');

        defaultFilter.schedulerList.set(filterValues);
        defaultFilter.schedulerList.saveFilterState();
    },

    onFormAccept() {
        const self = this;

        setTimeout(() => {
            self.onApply();
        }, 0);
    },

    onClose() {
        this.stopListening(this, FormAcceptBehavior.EVENTS.ACCEPT_EVENT);
    }
});

module.exports = SchedulersListFilterView;
