const React = require('react');
const ReactDOM = require('react-dom');

const app = require('../../../app');
const router = require('../../../router');
const SchedulersCollection = require('../../../collection/SchedulersCollection');
const SchedulerRemoveConfirmView = require('../../schedulers/modals/SchedulerRemoveConfirmView');
const SchedulerControllerBehavior = require('../../behaviors/controllers/SchedulerControllerBehavior');
const removedSchedulerNotification = require('../../components/infoPanels/tpl/removedSchedulerNotification.hbs');
const SingleTaskControlSectionLayout = require('../singleTask/SingleTaskControlSectionLayout');
const SingleSchedulerSecNavigation = require('../../navigation/secnavigation/SingleSchedulerSecNavigation');
const updateControls = require('../../utils/controls');

const clipboardClientMixin = require('../../components/clipboardClient/ClipboardClientMixin');
const Tags = require('../../../components/Task/Tags/Tags');

/**
 * @class SingleSchedulerControlSectionLayout
 * @extends SingleTaskControlSectionLayout
 *
 * @property {SchedulerModel} model
 */
const SingleSchedulerControlSectionLayout = SingleTaskControlSectionLayout.extend({

    template: require('./tpl/SingleSchedulerControlSectionLayout.hbs'),

    ui: _.extend({}, SingleTaskControlSectionLayout.prototype.ui, {
        disable: '.validate_disable',
        stop: '.execute_trigger_stop',
        cancel: '.cancel_trigger',
        tags: '#tags'
    }),

    events: _.extend({}, SingleTaskControlSectionLayout.prototype.events, {
        'click @ui.cancel': 'onCancel',
        'click @ui.stop': 'onExec'
    }),

    behaviors: {
        SchedulerControllerBehavior: {
            behaviorClass: SchedulerControllerBehavior.extend({
                onRemove() {
                    const view = this.view;

                    (new SchedulerRemoveConfirmView({
                        model: view.model,
                        onApprove() {
                            const backUrl = (String(window.location.href).replace(window.location.origin, ''));

                            SchedulersCollection.batchAction(
                                SchedulersCollection.BATCH_OPERATIONS.DELETE,
                                [this.model]
                            );

                            app.addNotification({
                                message: removedSchedulerNotification({
                                    id: this.model.get('id'),
                                    backUrl
                                })
                            });

                            this.close();

                            require('../../../views/layouts/BasicLayout').showNotifications();

                            view.onRemove.bind(view)();
                        }
                    })).show();
                }
            })
        }
    },

    initialize(options) {
        this.renderTags = this.renderTags.bind(this);
        this.onTagSet = this.onTagSet.bind(this);
        this.onTagDelete = this.onTagDelete.bind(this);

        this.options = _.extend({}, this.options, options);

        if (this.model.canBeEdited()) {
            this.listenTo(
                this.model.getTask(),
                'task:validating',
                updateControls(this.ui.disable, this.options.disableclass)
            );

            this.listenTo(this.model.getTask(), 'change:tags', this.renderTags);
        }
    },

    onCancel() {
        app.dropDraftScheduler();
        router.trigger('route:scheduler', this.model.get('id'), 'view');
    },

    onRender() {
        this.secnavigation.close();
        this.secnavigation.show(new SingleSchedulerSecNavigation({
            model: this.options.coreApp,
            layout: this.options.layout
        }));

        clipboardClientMixin.initializeClipboardClient(
            '.id-proxy-copier',
            'Scheduler ID has been copied to clipboard.'
        );

        this.renderTags();
    },

    renderTags() {
        if (this.ui.tags[0]) {
            const task = this.model.getTask();
            const rights = app.request('DRAFT_SCHEDULER') === null ? 'read' : 'write';

            ReactDOM.render(
                <Tags
                    id={this.model.get('id')}
                    combinedTags={task.get('tags')}
                    tags={task.get('tags') || []}
                    rights={rights}
                    namespace={'/schedulers/?tags=%%'}
                    upload={this.onTagSet}
                    remove={this.onTagDelete}
                    />,
                this.ui.tags[0]
            );
        }
    },

    onTagSet(data) {
        const task = this.model.getTask();

        task.set('tags', data.tags);
    },

    onTagDelete(data) {
        const task = this.model.getTask();
        const tags = task.get('tags');
        const index = tags.indexOf(data.tag);

        if (data.tag && index !== -1) {
            const newTags = [].concat(tags.slice(0, index), tags.slice(index + 1));

            task.set('tags', newTags);
        }
    },

    onRemove() {
        this.generateTargetUrl('/schedulers').done(generatedUrl => {
            router.navigate(generatedUrl, { trigger: true });
        });
    },

    serializeData() {
        const selfType = this.model.getType();
        const description = this.options.taskProps.types.where({ type: selfType });

        return {
            id: this.model.get('id'),
            type: selfType,
            status: this.model.get('status').toLowerCase(),
            description: this.model.get('description'),
            canBeStopped: this.model.canBeStopped(),
            canBeClonned: this.model.canBeClonned(),
            canBeStarted: this.model.canBeStarted(),
            canBeStartedOnce: this.model.canBeStartedOnce(),
            canBeEdited: this.model.canBeEdited(),
            typeDescription: (description[0] ? description[0].get('description') : ''),
            isEdit: app.request('DRAFT_SCHEDULER') === null
        };
    },

    onClose() {
        const task = this.model.getTask();

        this.stopListening(task, 'task:validating');
        this.stopListening(task, 'change:tags');
    }
});

module.exports = SingleSchedulerControlSectionLayout;
