const React = require('react');
const ReactDOM = require('react-dom');
const Provider = require('react-redux').Provider;
const store = require('../../../store/initializedStore');

/* eslint max-statements: [1, 14] */

const app = require('../../../app');
const router = require('../../../router');
const resources = require('../../../resources');
const FilterModel = require('../../../model/filter/FilterModel');
const TasksCollection = require('../../../collection/TasksCollection');
const SingleTaskActionErrorModalView = require('./SingleTaskActionErrorModalView');
const TaskRemoveConfirmView = require('../../tasks/taskRemoveConfirm/TaskRemoveConfirmView');
const TaskControllerBehavior = require('../../behaviors/controllers/TaskControllerBehavior');
const SingleTaskSecNavigation = require('../../navigation/secnavigation/SingleTaskSecNavigation');
const SingleTaskReleaseModalView = require('../../singleTask/SingleTaskReleaseModalView');
const SingleItemViewControlSectionLayout = require('../SingleItemViewControlSectionLayout');
const updateControls = require('../../utils/controls');

const removedTaskNotification = require('../../../views/components/infoPanels/tpl/removedTaskNotification.hbs');

const LinkModel = require('../../../components/LinkToCode/LinkModel');
const LinkToCode = require('../../../components/LinkToCode');
const Tags = require('../../../components/Task/Tags');

const clipboardClientMixin = require('../../components/clipboardClient/ClipboardClientMixin');

/**
 * @class SingleTaskControlSectionLayout
 * @extends SingleItemViewControlSectionLayout
 *
 * @property {TaskModel} model
 * @property {Marionette.Region} secnavigation
 */
const SingleTaskControlSectionLayout = SingleItemViewControlSectionLayout.extend({

    className: 'section__title',

    template: require('./tpl/SingleTaskCompactControlSectionLayout.hbs'),

    ui: _.extend({}, SingleItemViewControlSectionLayout.prototype.ui, {
        release: '.release_btn',
        disable: '.validate_disable',
        importantTrigger: '.important_trigger',
        linksToSources: '#links_to_sources',
        tags: '#tags'
    }),

    events: _.extend({}, SingleItemViewControlSectionLayout.prototype.events, {
        'click @ui.release': 'onRelease',
        'click @ui.importantTrigger': 'onImportantToggle'
    }),

    options: {
        disableclass: 'button_disable'
    },

    behaviors: {
        TaskControllerBehavior: {
            behaviorClass: TaskControllerBehavior.extend({
                onRemove() {
                    const view = this.view;

                    this.view.lastUserFilter = resources.userLastFilter({ filterModel: FilterModel });

                    (new TaskRemoveConfirmView({
                        model: view.model,
                        onApprove() {
                            const self = this;
                            const backUrl = (String(window.location.href).replace(window.location.origin, ''));

                            TasksCollection.batchAction(
                                TasksCollection.BATCH_OPERATIONS.DELETE,
                                [this.model]
                            ).done(() => {
                                app.addNotification({
                                    message: removedTaskNotification({
                                        id: self.model.get('id'),
                                        backUrl
                                    })
                                });

                                require('../../../views/layouts/BasicLayout').showNotifications();
                                view.onRemove.bind(view)();
                            });

                            this.close();
                        }
                    })).show();
                }
            })
        }
    },

    modelEvents: {
        'change:_duplicating': 'render',
        'change:status': 'render',
        'change:tags': 'renderTags'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);
        this.model = this.options.coreApp.getTask();

        this.listenTo(
            this.model,
            'task:validating',
            updateControls(this.ui.disable, this.options.disableclass)
        );

        this._linkModel = new LinkModel({
            type: this.model.get('type')
        });

        this._linkModel.fetch({
            success(model) {
                model.set('hasError', false);
            },
            error(model) {
                model.set('hasError', true);
            }
        });

        this.model.fetchContext().done(data => {
            this._taskVersion = data.tasks_version;
            this.render();
        });

        this.handleTagsChange = this.handleTagsChange.bind(this);
    },

    showActionError(errOpts) {
        (new SingleTaskActionErrorModalView(errOpts)).show();
    },

    onRender() {
        this.secnavigation.close();
        this.secnavigation.show(new SingleTaskSecNavigation({
            model: this.options.coreApp,
            layout: this.options.layout
        }));

        if (this.ui.linksToSources[0]) {
            ReactDOM.render(
                <LinkToCode model={this._linkModel} revision={this._taskVersion}/>,
                this.ui.linksToSources[0]
            );
        }

        clipboardClientMixin.initializeClipboardClient(
            '.id-proxy-copier',
            'Task ID has been copied to clipboard.'
        );

        this.renderTags();
    },

    renderTags() {
        if (this.ui.tags[0]) {
            ReactDOM.render(
                <Provider store={store}>
                    <Tags
                        id={this.model.get('id')}
                        tags={this.model.get('tags')}
                        rights={this.model.get('rights')}
                        onTagsChange={this.handleTagsChange}
                        />
                </Provider>,
                this.ui.tags[0]
            );
        }
    },

    handleTagsChange(tags) {
        this.model.set('tags', tags);
    },

    onImportantToggle(evtObj) {
        evtObj.preventDefault();
        this.model.toggleImportant();
    },

    serializeData() {
        const serialized = _.clone(this.model.attributes);

        serialized.status = serialized.status.toLowerCase();
        serialized.canBeEdited = this.model.canBeEdited();
        serialized.canBeStopped = this.model.canBeStopped();
        serialized.canBeDeleted = this.model.canBeDeleted();
        serialized.canBeReleased = this.model.canBeReleasedWithMessage();
        serialized.canBeClonned = this.model.canBeClonned();
        serialized.canBeExecuted = this.model.canBeExecuted();
        serialized.canBeSuspended = this.model.canBeSuspended();
        serialized.typeDescription = this.getTaskTypeDescription();
        serialized.canPriorityBeIncreased = this.model.canPriorityBeIncreased();

        return serialized;
    },

    getTaskTypeDescription() {
        const description = this.options.taskProps.types.where({ type: this.model.get('type') });

        return (description[0] ? description[0].get('description') : '');
    },

    onRelease() {
        (new SingleTaskReleaseModalView({
            model: this.model,
            ownerGroups: this.options.taskProps.groups.convertToList()
        })).show();
    },

    onRemove() {
        this.generateTargetUrl('/tasks').done(generatedUrl => {
            router.navigate(generatedUrl, { trigger: true });
        });
    },

    generateTargetUrl(blankUrl) {
        blankUrl = blankUrl || '/tasks';
        const generated = $.Deferred(); // eslint-disable-line

        if (this.lastUserFilter) {
            this.lastUserFilter
                .done(defaultFilter => {
                    generated.resolve(blankUrl + '?' + router.buildQueryString(_.extend({
                        page: 1,
                        pageCapacity: 20
                    }, defaultFilter.serializeAsUrl())));
                }).fail(() => {
                    generated.resolve(blankUrl);
                });
        } else {
            generated.resolve(blankUrl);
        }

        return generated;
    },

    onClose() {
        this.stopListening(this.model, 'task:validating');
    }
});

module.exports = SingleTaskControlSectionLayout;
