const ResourcesCollection = require('../../../collection/TaskResourcesCollection');
const TimeLineControl = require('../../components/timeLineControl/TimeLineControl');
const notification = require('../../../notification');

/**
 * @class XmlrpcStatisticsControls
 * @extends Marionette.Layout
 *
 * @property {TaskResourceModel} model
 * @property {Marionette.Region} timeLine
 */
const XmlrpcStatisticsControls = Marionette.Layout.extend({

    className: 'section__ii statistics_controls',

    template: require('./tpl/xmlrpcStatisticsControls.hbs'),

    ui: {
        rawDate: '.rawDate',
        daysCounter: '.days_before',
        daysCounterInput: '.days_before input'
    },

    events: {
        'bemmy:refresh': 'onDaysCounterChange',
        'timeline:selected': 'onTimelineSelected'
    },

    modelEvents: {
        'change:id': 'mapModelToView'
    },

    initialize() {
        this.statDateBefore = 1;
        this.getStatSourceResource();
    },

    onRender() {
        this.$('.input').bemmyInput();

        if (this.timeLine) {
            this.timeLine.close();
            this.timeLine.show(new TimeLineControl());
        }
    },

    /**
     * @param {object} evtObj
     * @param {string} date
     */
    onTimelineSelected(evtObj, date) {
        const diff = moment().diff(moment(date, 'DD.MM.YYYY'), 'days');

        this.ui.daysCounterInput.val(diff);
        this.onDaysCounterChange();
    },

    getStatResourceCollection() {
        const resources = new ResourcesCollection([], { order: '-id' });

        resources.mapFilterQueryOptions({
            type: 'SANDBOX_API_STATISTICS',
            attrs: JSON.stringify({
                date: moment().subtract(this.statDateBefore, 'days').format('YYYY-MM-DD'),
                type: 'xmlrpc'
            })
        });

        return resources;
    },

    getStatSourceResource() {
        const self = this;
        const resources = this.getStatResourceCollection();

        return resources.update().done(() => {
            const targetResource = resources.at(0);

            if (targetResource) {
                self.model.set('id', targetResource.get('id'));
            } else {
                notification.message('Looks we have no statistics for that date. Please select another one.', {
                    reload: false
                });
            }
        });
    },

    onDaysCounterChange() {
        const value = parseInt(this.ui.daysCounterInput.val(), 10);

        if (_.isNumber(value)) {
            this.statDateBefore = value;
            this.getStatSourceResource();
        }
    },

    mapModelToView() {
        const serialized = this.serializeData();

        this.ui.rawDate.text(serialized.date);
        this.ui.daysCounterInput.val(serialized.before);
    },

    serializeData() {
        return {
            before: this.statDateBefore,
            date: moment().subtract(this.statDateBefore, 'days').format('DD MMMM YYYY'),
            id: this.model.get('id')
        };
    }
});

module.exports = XmlrpcStatisticsControls;
