const sanitize = require('sanitize-html');
const app = require('../../app');
const router = require('../../router');
const error = require('../../error');
const ResourceModel = require('../../model/ResourceModel');
const ResourceTypeAutocompleteView = require('../controlSection/filter/selects/autocomplete/TaskTypeAutocompleteView');
const ResourceOwnerAutocompleteView = require('../createTask/CreateTaskOwnerConfigView.js');
const ResourceListConfigView = require('./ResourceListConfigView.js');

const opts = {
    allowedTags: sanitize.defaults.allowedTags.concat(['style', 'input', 'span']),
    allowedAttributes: Object.assign(
        {},
        sanitize.defaults.allowedAttributes,
        {
            input: ['name', 'value', 'checked', 'style', 'type'],
            td: ['colspan', 'rowspan'],
            th: ['colspan', 'rowspan'],
            '*': ['style', 'color', 'bgcolor', 'title', 'valign', 'class']
        }
    )
};

/**
 * @class CreateResourceLayout
 * @extends BasicTaskLayout
 *
 * @property {TaskModel} model
 * @property {Marionette.Region} ownerConfig
 */
const CreateResourceLayout = Marionette.Layout.extend({

    className: 'section__i section__i__resource_layout',
    template: require('./tpl/CreateResourceLayout.hbs'),

    ui: {
        popup: '.popup',
        description: '.textarea_description',
        save: '.btn_save_task',
        disable: '.validate_disable',
        $form: '.box',
        $input: '.box input[type="file"]',
        $label: '.box label',
        $errorMsg: '.box .box__error span',
        $status: '.box .box__uploading',
        $restart: '.box .box__restart',

        // Validation staff
        descriptionPopup: '.description-block .popup',
        descriptionPopupText: '.description-block .popup__text',
        ownerPopup: '.owner-block .popup',
        ownerPopupText: '.owner-block .popup__text',
        resourceTypePopup: '.resource-block .popup',
        resourceTypePopupText: '.resource-block .popup__text'
    },

    events: {
        'click @ui.save': 'onSave',
        'bemmy:refresh @ui.description': 'onDescriptionChange'
    },

    options: {
        loadclass: 'button_load',
        disableclass: 'button_disable',
        typeSelectorPlaceholder: 'Select type for a new Resource',
        typeSelectorInputClass: 'new_resouce_type'
    },

    modelEvents: {
        'change:description'() {
            this.ui.descriptionPopup.removeClass('popup_visible');
        },
        'change:owner'() {
            this.ui.ownerPopup.removeClass('popup_visible');
        },
        'change:resourceType'() {
            this.ui.resourceTypePopup.removeClass('popup_visible');
        }
    },

    regions: {
        typeSelector: '.type_selector',
        ownerConfig: '.owner_config',
        paramsConfig: '.params_config'
    },

    initialize() {
        this.model = app.reqres.request('DRAFT_RESOURCE') || new ResourceModel();
    },

    onRender() {
        this.typeSelector.close();
        this.ownerConfig.close();
        this.paramsConfig.close();

        this.typeSelector.show(this.getTypeSelector());
        this.ownerConfig.show(this.getOwnerSelector());
        this.paramsConfig.show(this.getListSelector());

        this.initControls();
    },

    initControls() {
        this.$('.input').bemmyInput();
        this.$('.textarea').bemmyTextarea();
        this.$('.button').bemmyButton();
        this.$('select').bemmySelect();
        this.$('.check').bemmyCheck();

        this.initDragNDrop();
    },

    initDragNDrop() {
        const self = this;

        self.ui.$form.each(() => {
            let	droppedFiles = false;
            const	showFiles = function (files) {
                const filename = files && files.length && files[0].name;

                self.ui.$label.append(filename);
                self.model.set('file', files[0]);
            };

            self.ui.$form.append('<input type="hidden" name="ajax" value="1" />');

            self.ui.$input.on('change', e => {
                showFiles(e.target.files);
            });

            self.ui.$form
                .addClass('has-advanced-upload')
                .on(
                    'drag dragstart dragend dragover dragenter dragleave drop',
                    e => {
                        e.preventDefault();
                        e.stopPropagation();
                    }
                )
                .on('dragover dragenter', () => {
                    self.ui.$form.addClass('is-dragover');
                })
                .on('dragleave dragend drop', () => {
                    self.ui.$form.removeClass('is-dragover');
                })
                .on('drop', e => {
                    droppedFiles = e.originalEvent.dataTransfer.files;
                    showFiles(droppedFiles);
                });

            self.ui.$restart.on('click', e => {
                e.preventDefault();

                self.ui.$form.removeClass('is-error is-success');
                self.ui.$input.trigger('click');
            });

            self.ui.$input.on('focus', () => {
                self.ui.$input.addClass('has-focus');
            })
                .on('blur', () => {
                    self.ui.$input.removeClass('has-focus');
                });
        });
    },

    _stringifyObject(obj) {
        const keys = Object.keys(obj);

        const result = keys.reduce((result, item) => {
            result.push(item + '=' + obj[item]);

            return result;
        }, []);

        return result.length ? result.join(',') : '';
    },

    validated() {
        const errorMessages = this.model.validate();

        if (errorMessages.length) {
            errorMessages.forEach(error => {
                this.ui[error.field + 'PopupText'].text(error.message);
                this.ui[error.field + 'Popup'].toggleClass('popup_visible', true);
            });

            return false;
        }

        return true;
    },

    onSave() {
        if (this.validated()) {
            const self = this;
            const formData = new FormData();

            self.ui.save.addClass(self.options.loadclass);

            if (self.ui.$form.hasClass('is-uploading')) {
                return false;
            }

            self.ui.$form.addClass('is-uploading').removeClass('is-error');

            const attributes = this.model.toJSON();

            Object.keys(attributes).forEach(key => {
                const value = attributes[key];

                formData.append(
                    key,
                    value !== null && typeof value === 'object' && !(value instanceof File) ?
                        self._stringifyObject(value) :
                        value
                );
            });

            $.ajax({
                type: 'POST',
                url: this.model.url(),
                xhr() {
                    const xhr = new window.XMLHttpRequest();

                    xhr.addEventListener('progress', evt => {
                        if (evt.lengthComputable) {
                            const percentComplete = evt.loaded / evt.total;

                            self.ui.$status.text(Math.round(percentComplete * 100) + '%');
                        }
                    }, false);

                    return xhr;
                },
                xhrFields: {
                    withCredentials: true
                },
                crossDomain: true,
                data: formData,
                processData: false,
                contentType: false
            })
                .complete(() => {
                    self.ui.$form.removeClass('is-uploading');
                    self.ui.save.removeClass(self.options.loadclass);
                })
                .success(data => {
                    self.ui.$form.addClass(data.id ? 'is-success' : 'is-error');

                    if (data.id) {
                        router.navigate('/resource/' + data.id + '/view', { trigger: true });
                    } else {
                        self.ui.$errorMsg.text(data.error);
                    }

                    self.ui.save.removeClass(self.options.loadclass);
                })
                .error(xhr => {
                    error.fromXHR(xhr);
                    self.ui.$form.addClass('is-error');
                    self.ui.save.removeClass(self.options.loadclass);
                });
        }
    },

    onModelChange() {
        app.setDraftTask(this.getTask());
    },

    onDescriptionChange() {
        this.model.set('description', sanitize(this.ui.description.find('textarea').val(), opts));
    },

    getTask() {
        return this.model;
    },

    getTypeSelector() {
        return new ResourceTypeAutocompleteView({
            model: this.model,
            items: this.options.types,
            value: this.model.get('resourceType'),
            placeHolder: this.options.typeSelectorPlaceholder,
            modelTargetKey: 'resourceType',
            selectClassName: this.options.typeSelectorInputClass
        });
    },

    getOwnerSelector() {
        return new ResourceOwnerAutocompleteView({
            groups: this.options.user.get('groups'),
            model: this.model,
            user: this.options.user
        });
    },

    getListSelector() {
        return new ResourceListConfigView({
            model: this.model,
            modelTargetKey: 'attrs'
        });
    }
});

module.exports = CreateResourceLayout;
