const _ = require('underscore');

/* eslint max-len: [1, 160] */
/* eslint max-statements: [1, 11] */

const app = require('../../app');
const info = require('../../info');
const error = require('../../error');
const router = require('../../router');
const TaskModel = require('../../model/TaskModel');
const CreateTaskLayout = require('../createTask/CreateTaskLayout');
const SchedulerConfigView = require('./SchedulerConfigView');
const SchedulerAuthorConfigView = require('./SchedulerAuthorConfigView');
const SchedulersCollection = require('../../collection/SchedulersCollection');
// eslint-disable-next-line no-unused-vars
const notification = require('../../notification');
const updateControls = require('../utils/controls');

/**
 * @class CreateSchedulerLayout
 * @extends CreateTaskLayout
 *
 * @property {SchedulerModel} model
 */
const CreateSchedulerLayout = CreateTaskLayout.extend({

    template: require('./tpl/CreateSchedulerLayout.hbs'),
    className: 'section__i section__i-edit',

    ui: _.extend({}, CreateTaskLayout.prototype.ui, {
        runOnce: '.btn_exec_once_task',
        cancel: '.btn_cancel_task',
        disable: '.validate_disable'
    }),

    events: _.extend({}, CreateTaskLayout.prototype.events, {
        'click @ui.runOnce': 'onRunOnce',
        'click @ui.cancel': 'onCancel'
    }),

    modelEvents: _.extend({}, CreateTaskLayout.prototype.modelEvents, {
        'scheduler:runOnce': 'onRunOnce'
    }),

    initialize(options) {
        CreateTaskLayout.prototype.initialize.bind(this)(options);

        this.onRunOnce = _.debounce(this.onRunOnce, 400).bind(this);

        this.listenTo(Backbone, 'scheduler:exec', this.onSchedulerExec);
        this.listenTo(Backbone, 'scheduler:exec:finished', this.onSchedulerExecFinished);
        this.listenTo(Backbone, 'scheduler:runOnce', this.onSchedulerRunOnce);
        this.listenTo(Backbone, 'scheduler:runOnce:finished', this.onSchedulerRunOnceExecFinished);
    },

    onSchedulerExec() {
        this.ui.exec.addClass('button_load');
    },

    onSchedulerExecFinished() {
        this.ui.exec.removeClass('button_load');
    },

    onSchedulerRunOnceExec() {
        this.ui.runOnce.addClass('button_load');
    },

    onSchedulerRunOnceExecFinished() {
        this.ui.runOnce.removeClass('button_load');
    },

    showRegions() {
        CreateTaskLayout.prototype.showRegions.apply(this, arguments);

        this.schedulerConfig.close();
        this.schedulerConfig.show(this.getSchedulerConfigView());
    },

    bindTaskEvents() {
        const task = this.getTask();

        this.listenTo(this.model, 'change', this.onModelChange);
        this.listenTo(task, 'change:owner', this.onTaskOwnerChange);
        this.listenTo(task, 'change:author', this.onTaskAuthorChange);
        this.listenTo(task, 'invalid:description', this.onValidationFail);
        this.listenTo(task, 'invalid:kill_timeout', this.onValidationFail);
        this.listenTo(task, 'invalid:disk_space', this.onValidationFail);

        if (this.model.canBeEdited()) {
            this.listenTo(task, 'task:validating', updateControls(this.ui.disable, this.options.disableclass));
        }
    },

    onModelChange() {
        app.setDraftScheduler(this.model);
    },

    onTaskOwnerChange() {
        this.model.set('owner', this.getTask().get('owner'));
    },

    onTaskAuthorChange() {
        this.model.set('author', this.getTask().get('author'));
    },

    getOwnerConfigView() {
        return new SchedulerAuthorConfigView({
            groups: this.options.taskProps.groups,
            model: this.getTask(),
            user: this.options.user,
            author: this.model.get('author')
        });
    },

    getSchedulerConfigView() {
        return new SchedulerConfigView({
            model: this.model
        });
    },

    getTask() {
        return this.model.getTask();
    },

    onCancel() {
        app.dropDraftScheduler();
        router.trigger('route:scheduler', this.model.get('id'), 'view');
    },

    onExec() {
        const self = this;

        Backbone.trigger('scheduler:exec');
        this.ui.exec.addClass(this.options.loadclass);

        return this.validateAndSave()
            .done(() => {
                const schedulerTime = self.model.get('time');
                const schedule = self.model.get('schedule');

                if (((schedulerTime.next && moment(schedulerTime.next).diff(moment(), 'seconds') <= 0)) || schedule.repetition) {
                    SchedulersCollection.batchAction(
                        SchedulersCollection.BATCH_OPERATIONS.START,
                        [self.model]
                    ).done(data => {
                        Backbone.trigger('scheduler:exec:finished');

                        if (data && data[0] && data[0].id) {
                            app.dropDraftScheduler();
                            router.trigger('route:scheduler', data[0].id, 'view');
                        }

                        app.addNotification({
                            message: (require('../components/globalMessage/tpl/schedulerSavedNotificationMessage.hbs'))({ id: self.model.get('id') })
                        });
                        info.message(
                            (require('../components/globalMessage/tpl/schedulerSavedNotificationMessage.hbs'))({ id: self.model.get('id') }),
                            { reload: false }
                        );
                    });
                } else {
                    // Next run is set to future
                    Backbone.trigger('scheduler:exec:finished');

                    app.addNotification({
                        message: (require('../components/globalMessage/tpl/schedulerDelayedNotificationMessage.hbs'))({ id: self.model.get('id') })
                    });
                    info.message(
                        (require('../components/globalMessage/tpl/schedulerDelayedNotificationMessage.hbs'))({ id: self.model.get('id') }),
                        { reload: false }
                    );
                }
            })
            .always(() => {
                Backbone.trigger('scheduler:exec:finished');

                if (self.ui.exec && self.ui.exec.removeClass) {
                    self.ui.exec.removeClass(self.options.loadclass);
                }
            });
    },

    onRunOnce() {
        const self = this;
        const scheduler = self.model;
        const task = scheduler.task;

        Backbone.trigger('scheduler:runOnce');
        this.ui.runOnce.addClass(this.options.loadclass);

        return self
            .validateAndSave()
            .done(() => {
                /* eslint camelcase: 0 */

                task.save({
                    scheduler_id: scheduler.id,
                    type: null
                })
                    .then(() => {
                        const status = task.get('status');
                        const action = (status === TaskModel.STATUS.WAITING || status === TaskModel.STATUS.WATCHING ? 'stop' : 'start');
                        const TasksCollection = require('../../collection/TasksCollection');

                        Backbone.trigger('scheduler:runOnce:finished');

                        TasksCollection.batchAction(action, [task]).done(() => {
                            router.navigate('/task/' + task.get('id') + '/view', { trigger: true });
                        });
                    })
                    .fail(e => {
                        Backbone.trigger('scheduler:runOnce:finished');

                        self.showNotValidMsg(e);
                    });
            })
            .fail(e => {
                Backbone.trigger('scheduler:runOnce:finished');

                self.showNotValidMsg(e);
            })
            .always(() => {
                Backbone.trigger('scheduler:runOnce:finished');

                self.ui.runOnce.removeClass(self.options.loadclass);
            });
    },

    onSave() {
        const self = this;

        this.ui.exec.addClass(this.options.loadclass);

        return this.validateAndSave()
            .done(() => {
                app.dropDraftScheduler();
                router.trigger('route:scheduler', self.model.get('id'), 'view');
            })
            .fail(e => {
                self.showNotValidMsg(e);
            })
            .always(() => {
                if (self.options.loadclass && self.ui && self.ui.exec && self.ui.exec.removeClass) {
                    self.ui.exec.removeClass(self.options.loadclass);
                }
            });
    },

    /**
     * @description Validates all the custom fields of the task and tries to save it.
     * @returns {JQueryDeferred|Boolean} jqXHR .save() result or FALSE if custom fields validation failed
     */
    validateAndSave() {
        const self = this;
        const saved = $.Deferred(); // eslint-disable-line
        const generalFieldsErrors = this.model.validateGeneralFields();

        if (generalFieldsErrors.length === 0) {
            this.validateCustomFields()
                .done(() => {
                    self.model
                        .save(null, { wait: false, parse: false })
                        .done(() => {
                            saved.resolve();
                            self.model.setAsDraft();
                        })
                        .fail(xhr => {
                            saved.reject();
                            error.fromXHR(xhr);
                        });
                })
                .fail(() => {
                    saved.reject();
                });
        } else {
            saved.reject();
        }

        return saved;
    },

    serializeData() {
        return _.extend({
            canBeStarted: this.model.canBeStarted(),
            canBeStartedOnce: this.model.canBeStartedOnce(),
            canBeStopped: this.model.canBeStopped(),
            canBeEdited: this.model.canBeEdited()
        }, this.model.serialize());
    },

    onClose() {
        this.stopListening(this.model, 'change');

        const task = this.getTask();

        this.stopListening(task, 'change:owner');
        this.stopListening(task, 'change:author');
        this.stopListening(task, 'invalid:description');
        this.stopListening(task, 'invalid:kill_timeout');
        this.stopListening(task, 'invalid:disk_space');

        if (this.model.canBeEdited()) {
            this.stopListening(task, 'task:validating');
        }
    }
});

module.exports = CreateSchedulerLayout;
