const UserGroupModel = require('../../model/resources/UserGroupModel');

/**
 * @class CreateTaskAuthorConfigView
 * @extends Marionette.ItemView
 *
 * @property {TaskModel} model
 */
const CreateTaskAuthorConfigView = Marionette.ItemView.extend({

    tagName: 'td',
    className: 'c',

    template: require('./tpl/SchedulerAuthorConfigView.hbs'),

    options: {
        user: null
    },

    ui: {
        ownerSelect: '.owner select',
        authorSelect: '.author select',
        remember: '.check',
        popupErrorOwner: '.popup-error-owner',
        popupErrorAuthor: '.popup-error-author',
        popupErrorOwnerText: '.popup-error-owner .popup__text',
        popupErrorAuthorText: '.popup-error-author .popup__text'
    },

    events: {
        'change @ui.ownerSelect': 'onChangedOwnerFilter',
        'change @ui.authorSelect': 'onChangedAuthorFilter'
    },

    modelEvents: {
        'invalid:owner': 'onValidationOwnerError',
        'invalid:author': 'onValidationAuthorError',
        sync: 'onTaskSaved'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);

        this.setDefaultUser();

        this.updateAuthorFilter(this.model.get('owner'));
    },

    setDefaultUser() {
        let owner = this.model.get('owner');

        if (!owner) {
            const groups = this.options.user.getGroups().convertToList();

            owner = this.options.user.getPreferences().get('newTaskOwner');

            if (!owner && groups.length === 1) {
                owner = groups[0];
            }

            this.model.set('owner', owner);
        }
    },

    onRender() {
        this.$('.select').bemmySelect();
        this.$('.check').bemmyCheck();
    },

    onChangedOwnerFilter() {
        const owner = this.ui.ownerSelect.val();

        this.ui.popupErrorOwner.removeClass('popup_visible');

        this.model.set('owner', owner);

        this.updateAuthorFilter(owner);
    },

    onChangedAuthorFilter() {
        const author = this.ui.authorSelect.val();

        this.ui.popupErrorAuthor.removeClass('popup_visible');

        this.model.set('author', author);
    },

    updateAuthorFilter(owner) {
        const self = this;
        let users;

        self.options.users = self.options.users || {};

        if (self.options.users[owner]) {
            users = self.options.users[owner];

            self.render();
        } else {
            new UserGroupModel({ name: owner })
                .fetch({ reset: true })
                .done(result => {
                    users = result.members.map(user => {
                        return user.get('name');
                    });

                    self.options.users[owner] = users;

                    self.render();
                })
                .fail(() => {
                    self.options.users[owner] = [self.options.user.get('login')];

                    self.render();
                });
        }
    },

    onValidationOwnerError(err) {
        this.ui.popupErrorOwnerText.text(err.message);
        this.ui.popupErrorOwner.addClass('popup_visible');
    },

    onValidationAuthorError(err) {
        this.ui.popupErrorAuthorText.text(err.message);
        this.ui.popupErrorAuthor.addClass('popup_visible');
    },

    onTaskSaved() {
        const owner = this.model.get('owner');

        if (owner &&
            owner !== this.options.user.get('login') &&
            owner !== 'OTHER' &&
            this.ui.remember.hasClass('checked')) {
            const prefs = this.options.user.getPreferences();

            prefs.set('newTaskOwner', owner);
        }
    },

    serializeData() {
        const owner = this.model.get('owner');
        const groups = this.options.user.isAdmin() ?
            this.options.groups.map(group => {
                return group.get('name');
            }) :
            this.options.user.getGroups().convertToList();
        const users = this.options.users || {};

        return {
            owner,
            author: this.options.author,
            rememberNewTaskOwner: true,
            user: {
                login: this.options.user.get('login'),
                groups
            },
            users: users[owner]
        };
    }
});

module.exports = CreateTaskAuthorConfigView;
