/* eslint camelcase: 0 */

// eslint-disable-next-line no-unused-vars
const Timer = require('../../helpers/utils/Timer');
const TimeInterval = require('../components/timeInterval/TimeInterval');
const CreateTaskLayout = require('../createTask/CreateTaskLayout');
const SchedulerWeeklyRepetitionView = require('./SchedulerWeeklyRepetitionView');

/**
 * @class SchedulerConfigView
 * @extends Marionette.ItemView
 *
 * @property {SchedulerModel} model
 * @property {Marionette.Region} retryInterval
 * @property {Marionette.Region} repetitionWeekly
 * @property {Marionette.Region} repetitionInterval
 */
const SchedulerConfigView = Marionette.Layout.extend({

    className: 'section__form',
    template: require('./tpl/SchedulerConfigView.hbs'),

    ui: {
        start: '.start',
        retry: '.retry',
        startDate: '.scheduler-start-date',
        startTime: '.scheduler-start-time',
        repetition: '.repetition',
        runSequential: '.run-sequential',
        startDateInput: '.scheduler-start-date input',
        startTimeInput: '.scheduler-start-time input',
        startDateTimeCnt: '.scheduler-start-datetime',
        retryIntervalCnt: '.scheduler-retry-interval',

        repetitionNoneTrigger: '.repetition-none-trigger',
        repetitionDaysCnt: '.scheduler-repetition-days',
        repetitionDaysTrigger: '.repetition-days-trigger',
        repetitionIntervalCnt: '.scheduler-repetition-interval',
        repetitionIntervalTrigger: '.repetition-interval-trigger',

        startDateTimePopup: '.scheduler-start-datetime .popup',
        startDateTimePopupText: '.scheduler-start-datetime .popup__text',
        retryPopup: '.scheduler-repetition-days .popup',
        retryPopupText: '.scheduler-repetition-days .popup__text'
    },

    events: {
        'bemmy:set @ui.start': 'onStartModeChange',
        'bemmy:set @ui.retry': 'onRetryModeChange',
        'bemmy:set @ui.repetition': 'onRepetitionModeChange',
        'bemmy:set .scheduler-repetition-days .check': 'onRepetitionDaysChange',
        'bemmy:set @ui.runSequential': 'onRunSequential',

        'bemmy:refresh @ui.startDate': 'onStartDateChange',
        'bemmy:refresh @ui.startTime': 'onStartTimeChange',
        'input @ui.startDateInput': 'onStartDateInput'
    },

    modelEvents: {
        'invalid:startDateTime': 'onValidationFailed',
        'invalid:retry': 'onValidationFailed',
        'change:schedule': 'onScheduleChange'
    },

    onRender() {
        this.$('.button').bemmyButton();
        this.$('.select').bemmySelect();
        this.$('.input').bemmyInput();
        this.$('.check').bemmyCheck();

        this.ui.startDate.bemmyDatepicker('', {});
        this.ui.startDate.datepicker(
            'setDate',
            moment.utc(this.model.get('schedule').start_time).local().format('DD.MM.YYYY')
        );
        this.ui.startDateInput.val(moment.utc(this.model.get('schedule').start_time).local().format('DD.MM.YYYY'));

        this.listenTo(this.ui.startDate.bemmy(), 'date', this._setStartDateChange);

        this.showRegions();
    },

    showRegions() {
        this.repetitionInterval.close();
        this.repetitionWeekly.close();
        this.retryInterval.close();

        this.repetitionInterval.show(this.getRepetitionInterval());
        this.repetitionWeekly.show(this.getRepetitionWeekly());
        this.retryInterval.show(this.getRetryInterval());
    },

    getRepetitionWeekly() {
        return new SchedulerWeeklyRepetitionView({
            model: this.model
        });
    },

    getRepetitionInterval() {
        return new TimeInterval({

            model: this.model,

            /** Will be call with model as this */
            modelSetter(interval) {
                this.setIntervalRepetition(interval);
            },

            /** Will be call with model as this */
            modelGetter() {
                const schedule = this.get('schedule');

                return (schedule && schedule.repetition && schedule.repetition.interval) ?
                    schedule.repetition.interval :
                    0;
            },

            controlWidth: 121,
            bindedModelKey: 'schedule'
        });
    },

    getRetryInterval() {
        return new TimeInterval({

            model: this.model,

            /** Will be call with model as this */
            modelSetter(interval) {
                this.setRetryInterval(interval);
            },

            /** Will be call with model as this */
            modelGetter() {
                const schedule = this.get('schedule');

                return (schedule && schedule.retry && schedule.retry.interval) ? schedule.retry.interval : '1h';
            },

            controlWidth: 105,
            bindedModelKey: 'schedule'
        });
    },

    onStartDateChange() {
        CreateTaskLayout.prototype.hideFieldValidateMessage.call(this, 'startDateTime');
        this.model.setNextRunDate(this.ui.startDateInput.val());
    },

    onStartDateInput() {
        CreateTaskLayout.prototype.hideFieldValidateMessage.call(this, 'startDateTime');
        this.model.setNonZeroBasedNextRunDate(moment(
            this.ui.startDateInput.val() + ' ' + this.ui.startTimeInput.val(),
            'DD.MM.YYYY HH:mm'
        ));
    },

    onScheduleChange() {
        const schedule = this.model.get('schedule');

        CreateTaskLayout.prototype.hideFieldValidateMessage.call(this, 'retry');

        if (schedule) {
            if (schedule.repetition === null) {
                this.ui.repetitionDaysCnt.addClass('hidden');
                this.ui.repetitionIntervalCnt.addClass('hidden');
            } else if (schedule.repetition.interval === undefined) {
                this.ui.repetitionIntervalCnt.addClass('hidden');
                this.ui.repetitionDaysCnt.removeClass('hidden');
            } else {
                this.ui.repetitionDaysCnt.addClass('hidden');
                this.ui.repetitionIntervalCnt.removeClass('hidden');
            }
        }
    },

    _setStartDateChange(rawDate) {
        const date = moment(rawDate);

        this.ui.startDateInput.val((date.date() < 10 ? '0' + date.date() : date.date()) + '.' +
            (date.month() < 10 ? '0' + date.month() : date.month()) + '.' +
            date.year());
        this.ui.startDate.trigger('bemmy:refresh');
    },

    onStartTimeChange() {
        CreateTaskLayout.prototype.hideFieldValidateMessage.call(this, 'startDateTime');
        this.model.setNextRunTime(this.ui.startTimeInput.val());
    },

    onValidationFailed(err) {
        CreateTaskLayout.prototype.showFieldValidateMessage.call(this, err.field, err.message);
    },

    onStartModeChange(evtObj, value) {
        if (value === 'immediately') {
            this.model.setNextRun();
            this.ui.startDateTimeCnt.addClass('hidden');
        } else {
            this.model.setNextRun(moment());
            this.render();
        }
    },

    onRepetitionModeChange(evtObj, value) {
        if (value === 'none') {
            this.model.setNoneRepetition();
        } else if (value === 'interval') {
            this.model.setIntervalRepetition(18000);
        } else {
            this.model.setWeeklyRepetition([0, 1, 2, 3, 4, 5, 6]);
        }
    },

    onRetryModeChange(evtObj, value) {
        /* eslint max-statements: [1, 11] */

        const schedule = this.model.get('schedule');

        schedule.fail_on_error = false;

        if (value === 'interval') {
            schedule.retry = {
                interval: 86400,
                ignore: false
            };

            this.ui.retryIntervalCnt.removeClass('hidden');
        } else {
            if (value === 'none') {
                schedule.retry = {
                    interval: 0,
                    ignore: true
                };
            } else if (value === 'failure') {
                schedule.fail_on_error = true;
                schedule.retry = {
                    interval: 0,
                    ignore: false
                };
            }

            this.ui.retryIntervalCnt.addClass('hidden');
        }

        this.model.setSchedule(schedule);
    },

    onRepetitionDaysChange() {
        const days = [];

        this.ui.repetitionDaysCnt.find('.checked').each((index, elt) => {
            days.push($(elt).find('input').val());
        });

        const schedule = this.model.get('schedule');

        schedule.repetition = {
            interval: null,
            weekly: days
        };

        this.model.setSchedule(schedule);
    },

    serializeRepetition() {
        const schedule = this.model.get('schedule');
        let serialized = false;

        if (schedule.repetition) {
            if (schedule.repetition.weekly && schedule.repetition.weekly.length) {
                serialized = { days: schedule.repetition.weekly };
            }

            if (schedule.repetition.interval) {
                serialized = {
                    measure: 'seconds',
                    interval: schedule.repetition.interval
                };
            }
        }

        return serialized;
    },

    serializeRetry() {
        const schedule = this.model.get('schedule');
        let serialized = {};

        if (schedule.retry && schedule.retry.ignore) {
            serialized = false;
        }

        if (typeof serialized !== 'boolean' && schedule.fail_on_error) {
            serialized.failure = true;
        }

        if (typeof serialized !== 'boolean' && schedule.retry && schedule.retry.interval) {
            serialized.interval = schedule.retry.interval;
        }

        return serialized;
    },

    onRunSequential(evtObj, runSequentially) {
        const schedule = this.model.get('schedule');

        schedule.sequential_run = runSequentially;

        this.model.setSchedule(schedule);
    },

    serializeNextRun() {
        const schedule = this.model.get('schedule');
        const momentified = (schedule.start_time ? moment.utc(schedule.start_time).local() : null);

        return (momentified ? {
            date: momentified.format('DD.MM.YYYY'),
            time: momentified.format('HH:mm')
        } : false);
    },

    serializeData() {
        const serialized = {
            startExecution: this.serializeNextRun(),
            sequentialRun: this.model.get('schedule').sequential_run,
            repetition: this.serializeRepetition(),
            retry: this.serializeRetry()
        };

        return serialized;
    }
});

module.exports = SchedulerConfigView;
