const CustomFieldModel = require('../../model/CustomFieldModel');
const ResourceCustomFieldView = require('./customProps/ResourceCustomFieldView');
const CreateTaskDNSConfigView = require('./CreateTaskDNSConfigView');
const CreateTaskHostsConfigView = require('./CreateTaskHostsConfigView');
const CreateTaskSemaphoresConfigView = require('./CreateTaskSemaphoresConfigView');

/**
 * @class CreateTaskAdvanced
 * @extends Backbone.Marionette.Layout
 */
const CreateTaskAdvanced = Marionette.Layout.extend({

    className: 't__desc table_advanced_fields',

    template: require('./tpl/CreateTaskAdvanced.hbs'),

    ui: {
        toggler: '.link__toggler',

        ram: '.task_ram',
        ramdrive: '.task_ramdrive',
        cores: '.task_cores',
        clientTags: '.task_client-tags',

        isDoNotRestart: '.is-do-not-restart',
        isFailOnAnyError: '.fail-on-any-error',
        isHidden: '.is-hidden',
        isPrivileged: '.is-privileged',

        ramPopup: '.disk_ram_cnt .popup',
        ramPopupText: '.disk_ram_cnt .popup__text',
        ramdrivePopup: '.disk_ramdrive_cnt .popup',
        ramdrivePopupText: '.disk_ramdrive_cnt .popup__text',
        coresPopup: '.cores_cnt .popup',
        coresPopupText: '.cores_cnt .popup__text',
        clientTagsPopup: '.client-tags_cnt .popup',
        clientTagsPopupText: '.client-tags_cnt .popup__text'
    },

    events: {
        'click @ui.toggler': 'onToggleClickHandler',

        'change @ui.ram': 'onRamChange',
        'change @ui.ramdrive': 'onRamDriveChange',
        'change @ui.cores': 'onCoresChange',
        'change @ui.clientTags': 'onClientTagsChange',

        'bemmy:set @ui.isPrivileged': 'onPrivilegedChange',
        'bemmy:set @ui.isHidden': 'onHiddenChange',
        'bemmy:set @ui.isDoNotRestart': 'onRestartChange',
        'bemmy:set @ui.isFailOnAnyError': 'onAnyErrorFailChange'
    },

    options: {
        descCntOpenedClass: 't__desc_open'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);

        this.listenTo(this.model, 'invalid:ram', this.onValidationFail);
        if (!this.options.isScheduler) {
            this.listenTo(this.model, 'invalid:ramdrive', this.onValidationFail);
        }
        this.listenTo(this.model, 'invalid:cores', this.onValidationFail);
        this.listenTo(this.model, 'invalid:clientTags', this.onValidationFail);
    },

    initControls() {
        this.$('.input').bemmyInput();
        this.$('.button').bemmyButton();
        this.$('.select').bemmySelect();
        this.$('.check').bemmyCheck();
        this.$('.textarea').bemmyTextarea();
    },

    onRender() {
        this.initControls();

        this.initContainerResource();
        this.initArchiveResource();
        this.initTaskDNSConfig();
        this.initTasksHostsConfig();

        if (!this.options.isScheduler) {
            this.initTasksSemaphoresConfig();
        }
    },

    isContainerResourceExists() {
        let containerCustomProp = this.model.get('customFields').filter(field => {
            return field.type === 'container';
        });

        containerCustomProp = containerCustomProp && containerCustomProp[0];

        return containerCustomProp;
    },

    isArchieveResourceExists() {
        return Boolean(this.model);
    },

    initTaskDNSConfig() {
        this.dnsConfig.close();
        this.dnsConfig.show(new CreateTaskDNSConfigView({
            model: this.model,
            dns: this.options.taskProps.dns
        }));
    },

    initTasksHostsConfig() {
        const taskProps = this.options.taskProps;

        this.hostsConfig.close();
        this.hostsConfig.show(new CreateTaskHostsConfigView({
            cpuModels: taskProps.cpuModels,
            platforms: taskProps.platforms,
            hosts: taskProps.hosts,
            model: this.model
        }));
    },

    initTasksSemaphoresConfig() {
        this.semaphoresConfig.close();
        this.semaphoresConfig.show(new CreateTaskSemaphoresConfigView({
            model: this.model,
            taskProps: this.options.taskProps,
            collection: this.model.get('semaphores')
        }));
    },

    initContainerResource() {
        const containerCustomProp = this.isContainerResourceExists();

        if (containerCustomProp) {
            const containerConf = {
                context: containerCustomProp.context || {
                    multiple: false,
                    types: ['LXC_CONTAINER']
                },
                description: '',
                name: containerCustomProp.name,
                required: Boolean(containerCustomProp.required),
                title: containerCustomProp.title || 'Container',
                type: 'resource',
                value: containerCustomProp.value,
                parentModel: this.model
            };
            const containerModel = new CustomFieldModel(containerConf);

            this.containerControl.close();
            this.containerControl.show(new ResourceCustomFieldView({
                model: containerModel,
                taskModel: this.model,
                taskProps: this.options.taskProps
            }));
        }
    },

    initArchiveResource() {
        const isArchieveResourceExists = this.isArchieveResourceExists();

        if (isArchieveResourceExists) {
            const achieveResourceConf = {
                context: {
                    multiple: false,
                    types: ['SANDBOX_TASKS_ARCHIVE']
                },
                title: 'Tasks archive resource',
                type: 'resource',
                value: this.model.get('tasks_archive_resource'),
                validateHelper: null
            };
            const achieveResourceModel = new CustomFieldModel(_.extend(
                CustomFieldModel.parse(achieveResourceConf),
                { parentModel: this.model }
            ));

            this.listenTo(achieveResourceModel, 'change:value', this.onChangeAchieveResource);

            this.archiveResourceControl.close();
            this.archiveResourceControl.show(new ResourceCustomFieldView({
                model: achieveResourceModel,
                taskModel: this.model,
                taskProps: this.options.taskProps
            }));
        }
    },

    onHiddenChange(evtObj, isHidden) {
        this.model.set('hidden', isHidden);
    },

    onPrivilegedChange(eventObj, isPrivileged) {
        const task = this.model;
        const requirements = Object.assign({}, task.get('requirements'), { privileged: isPrivileged });

        task.set('requirements', requirements);
    },

    onRestartChange(evtObj, canNotBeRestarted) {
        this.model.set('doNotRestart', canNotBeRestarted);
    },

    onAnyErrorFailChange(evtObj, failOnAnyError) {
        this.model.set('fail_on_any_error', failOnAnyError);
    },

    onRamChange(evtObj) {
        const task = this.model;
        const requirements = task.get('requirements');

        this.hideFieldValidateMessage('ram');

        task.validateRamSpace({ ram: evtObj.target.value });

        requirements.ram = numeral().unformat(evtObj.target.value);

        task.set('requirements', requirements);
    },

    onRamDriveChange(evtObj) {
        const task = this.model;
        const requirements = task.get('requirements');

        this.hideFieldValidateMessage('ramdrive');

        task.validateRamDriveSpace({ ramdrive: evtObj.target.value });

        requirements.ramdrive = numeral().unformat(evtObj.target.value);

        task.set('requirements', requirements);
    },

    onCoresChange(evtObj) {
        const task = this.model;
        const requirements = task.get('requirements');

        this.hideFieldValidateMessage('cores');

        task.validateCores({ cores: evtObj.target.value });

        requirements.cores = numeral().unformat(evtObj.target.value);

        task.set('requirements', requirements);
    },

    onClientTagsChange(evtObj) {
        const task = this.model;
        const requirements = task.get('requirements');

        this.hideFieldValidateMessage('clientTags');

        task.validateClientTags({client_tags: evtObj.target.value}); // eslint-disable-line

        requirements.client_tags = evtObj.target.value;              // eslint-disable-line

        task.set('requirements', requirements);
    },

    onToggleClickHandler(eventObj) {
        eventObj.preventDefault();

        this.$el.toggleClass(this.options.descCntOpenedClass);

        return false;
    },

    onChangeAchieveResource() {
        if (arguments[0]) {
            const parentModel = arguments[0].get('parentModel');
            const value = arguments[0].get('value') || null;

            if (parentModel) {
                parentModel.set('tasks_archive_resource', value);
            }
        }
    },

    onValidationFail(err) {
        this.showFieldValidateMessage(err.field, err.message);
    },

    showFieldValidateMessage(fieldName, message) {
        this.ui[fieldName + 'PopupText'].text(message);
        this.ui[fieldName + 'Popup'].toggleClass('popup_visible', true);
    },

    hideFieldValidateMessage(fieldName) {
        this.ui[fieldName + 'Popup'].toggleClass('popup_visible', false);
    },

    serializeData() {
        const model = this.model.toJSON();

        return Object.assign(
            model,
            {
                isScheduler: Boolean(this.options.isScheduler),
                requirements: Object.assign({}, model.requirements, {
                    ramdrive: model.requirements.ramdrive && model.requirements.ramdrive.size ?
                        model.requirements.ramdrive.size :
                        model.requirements.ramdrive
                })
            }
        );
    },

    onClose() {
        this.stopListening(this.model, 'invalid:ram');

        if (!this.options.isScheduler) {
            this.stopListening(this.model, 'invalid:ramdrive');
        }

        this.stopListening(this.model, 'invalid:cores');
        this.stopListening(this.model, 'invalid:clientTags');
    }
});

module.exports = CreateTaskAdvanced;

