const CustomFieldsCollection = require('../../collection/CustomFieldsCollection');
const CustomFieldModel = require('../../model/CustomFieldModel');
const CustomFieldView = require('./customProps/CustomFieldView');

/**
 * @class CreateTaskCustomFieldsView
 * @extends Backbone.Marionette.CompositeView
 *
 * @property {TaskModel} model
 */
const CreateTaskCustomFieldsView = Marionette.CompositeView.extend({

    className: 'section__form',

    itemViewContainer: 'table',

    template: require('./tpl/CreateTaskCustomFieldsView.hbs'),

    initialize(options) {
        this.itemViewOptions = {
            taskModel: this.model,
            taskProps: options.taskProps,
            fieldViewMode: 'edit'
        };

        this.createCustomFieldsCollection(options.excludeContainer, options.outputType);
    },

    getItemView(item) {
        const itemView = CustomFieldView.getView(item) || this.constructor;

        if (!itemView) {
            throw new Error('An `itemView` must be specified', 'NoItemViewError');
        }

        return itemView;
    },

    /**
     * Create custom fields section
     * @param {Boolean} [excludeContainer] Flag of excluded container type field
     * @param {Boolean} [outputType] Set items to collection of parameters with [outputType] type's value
     * @returns {void}
     */
    createCustomFieldsCollection(excludeContainer, outputType) {
        /* eslint max-statements: [1, 13] */

        let fields = this.model.get('customFields');
        const tempCollection = new CustomFieldsCollection();
        let lastBlock;

        // SANDBOX-4230 – разный ответ от ручки получения кастомных полей на локальном SB и продовом
        if (this.model.editMode) {
            fields = fields.filter(field => {
                return !field.output;
            });
        }

        if (fields) {
            fields.forEach(function (field) {
                /* eslint complexity: [1, 9] */
                /* eslint max-depth: [1, 4] */

                if (!excludeContainer || (excludeContainer && field.type !== 'container')) {
                    if (outputType === undefined) {
                        tempCollection.add(new CustomFieldModel(_.extend(CustomFieldModel.parse(field), { parentModel: this.model })));
                    } else {
                        if (field.type === 'block') { // eslint-disable-line
                            lastBlock = field;
                        } else if (field.output === undefined || field.output === outputType) {
                            if (lastBlock) {
                                tempCollection.add(new CustomFieldModel(_.extend(CustomFieldModel.parse(lastBlock), { parentModel: this.model })));

                                lastBlock = false;
                            }

                            tempCollection.add(new CustomFieldModel(_.extend(CustomFieldModel.parse(field), { parentModel: this.model })));
                        }
                    }
                }
            }, this);

            tempCollection.forEach(fieldModel => {
                const subs = fieldModel.get('subFields');
                const keys = Object.keys(subs);

                if (keys.length > 0) {
                    keys.forEach(subValue => {
                        if (subs[subValue]) {
                            const items = typeof subs[subValue] === 'string' ?
                                [subs[subValue]] :
                                subs[subValue] instanceof Array ?
                                    subs[subValue] : [];

                            items.forEach(subName => {
                                const subField = tempCollection.findWhere({ name: subName });

                                if (typeof subField !== 'undefined') {
                                    subField.setDependFrom(fieldModel, subValue);
                                }
                            });
                        }
                    });
                }
            }, this);

            this.collection = tempCollection;

            if (this.collection && this.collection.models[0]) {
                this.collection.models[0].setAvailable();
            }

            this.collection.fetchResourcesDescription();
        }
    },

    getCollection() {
        return this.collection;
    }
});

module.exports = CreateTaskCustomFieldsView;
