/* eslint camelcase: 0 */
/* eslint max-statements: [1, 17] */

const sanitize = require('sanitize-html');
const app = require('../../app');
const error = require('../../error');
const router = require('../../router');
const TaskModel = require('../../model/TaskModel');
const SchedulerModel = require('../../model/SchedulerModel');
const PriorityConfig = require('./priorityConfig/PriorityConfig');
const TasksCollection = require('../../collection/TasksCollection');
const BasicTaskLayout = require('../tasks/tasksGrid/row/BasicTaskLayout');
const CreateTaskNotifyConfig = require('./CreateTaskNotifyConfig');
// eslint-disable-next-line no-unused-vars
const NotificationModel = require('../../model/NotificationModel');
// eslint-disable-next-line no-unused-vars
const NotificationsCollection = require('../../collection/NotificationsCollection');
const CreateTaskAdvancedView = require('./CreateTaskAdvanced');
const CreateTaskOwnerConfigView = require('./CreateTaskOwnerConfigView');
const CreateTaskSeTagConfigView = require('./CreateTaskSeTagConfigView');
const CreateTaskCustomFieldsView = require('./CreateTaskCustomFieldsView');
const updateControls = require('../utils/controls');

const opts = {
    allowedTags: sanitize.defaults.allowedTags.concat(['style', 'input', 'span']),
    allowedAttributes: Object.assign(
        {},
        sanitize.defaults.allowedAttributes,
        {
            input: ['name', 'value', 'checked', 'style', 'type'],
            td: ['colspan', 'rowspan'],
            th: ['colspan', 'rowspan'],
            '*': ['style', 'color', 'bgcolor', 'title', 'valign', 'class']
        }
    )
};

/**
 * @class CreateTaskLayout
 * @extends BasicTaskLayout
 *
 * @property {TaskModel} model
 * @property {Marionette.Region} ownerConfig
 * @property {Marionette.Region} customProps
 * @property {Marionette.Region} seTagsConfig
 * @property {Marionette.Region} priorityConfig
 * @property {Marionette.Region} notificationsConfig
 * @property {Marionette.Region} advancedConfig
 */
const CreateTaskLayout = BasicTaskLayout.extend({

    className: 'section__i',
    template: require('./tpl/CreateTaskLayout.hbs'),

    ui: {
        exec: '.btn_exec_task',
        save: '.btn_save_task',
        disable: '.validate_disable',

        descr: '.task_descr textarea',
        space: '.task_space',
        timeToKill: '.task_time_to_kill',
        controlsCnt: '.section__ctrl',

        // Validation stuff
        kill_timeoutPopup: '.kill_timeout_cnt .popup',
        kill_timeoutPopupText: '.kill_timeout_cnt .popup__text',
        disk_spacePopup: '.disk_space_cnt .popup',
        disk_spacePopupText: '.disk_space_cnt .popup__text',
        descriptionPopup: '.descr_cnt .popup',
        descriptionPopupText: '.descr_cnt .popup__text'
    },

    events: {
        'change @ui.descr': 'onDescrChange',
        'change @ui.space': 'onSpaceChange',
        'change @ui.timeToKill': 'onKillTimeChange',
        'click @ui.exec': 'onExec',
        'click @ui.save': 'onSave'
    },

    options: {
        loadclass: 'button_load',
        disableclass: 'button_disable'
    },

    /** REGIONS DEFINED DIRECTLY IN THE TEMPLATE */

    initialize(options) {
        this.listenTo(Backbone, 'task:exec', () => this.onExec(true));

        this.options.isScheduler = options.isScheduler || false;
        this.options.user = options.user || {};
        this.options.taskProps = options.taskProps || {};
        this.options.controlsDefaultOffset = ($(window).innerHeight() + 1); // +1 to be pinned by default

        this.model.editMode = true;

        this.listenTo(
            this.model,
            'task:validating',
            updateControls(this.ui.disable, this.options.disableclass)
        );

        this.pinTaskControls = _.debounce(this.pinTaskControls, 100).bind(this);
    },

    onRender() {
        this.initControls();
        this.showRegions();
        this.bindTaskEvents();
    },

    initControls() {
        this.$('.input').bemmyInput();
        this.$('.button').bemmyButton();
        this.$('.select').bemmySelect();
        this.$('.check').bemmyCheck();
        this.$('.textarea').bemmyTextarea();
    },

    showRegions() {
        const self = this;
        const seTags = this.model.get('se_tags');

        this.ownerConfig.close();
        this.priorityConfig.close();
        this.notificationsConfig.close();

        this.ownerConfig.show(this.getOwnerConfigView());
        this.priorityConfig.show(this.getPriorityConfigView());

        if (seTags && seTags.length) {
            this.seTagsConfig.close();
            this.seTagsConfig.show(this.getSeTagConfigView());
        }

        this.notificationsConfig.show(this.getNotifyConfigView());

        $.when(
            self.getTask().fetchCustomParameters(TaskModel.CUSTOM_PAR_MODE.EDIT),
            app.getHelperManager().loadExternalHelpers()
        )
            .done(() => {
                if (self.customProps) {
                    self.customProps.close();
                    self.customProps.show(self.getCustomPropsView());
                }
                self.advancedConfig.close();
                self.advancedConfig.show(self.getAdvancedConfigView());
                self.initControlsPinning();
            })
            .fail(function () {
            /* eslint no-console: 0 */
                console.log('SANDBOX: error during loading custom parameters or helpers', arguments);
            });
    },

    bindTaskEvents() {
        const task = this.getTask();

        this.listenTo(task, 'change', this.onModelChange);
        this.listenTo(task, 'invalid:description', this.onValidationFail);
        this.listenTo(task, 'invalid:kill_timeout', this.onValidationFail);
        this.listenTo(task, 'invalid:disk_space', this.onValidationFail);
    },

    getTask() {
        return this.model;
    },

    onExec(isFromChannel) {
        const self = this;

        if (!isFromChannel) {
            Backbone.trigger('task:exec');
        }

        this.ui.exec.addClass(this.options.loadclass);

        return this.validateAndSave()
            .done(() => {
                const task = self.getTask();

                TasksCollection
                    .batchAction(TasksCollection.BATCH_OPERATIONS.START, [task])
                    .done(() => {
                        Backbone.trigger('task:exec:finished');

                        app.dropDraftTask();
                        if (self.ui.exec) {
                            self.ui.exec.removeClass(self.options.loadclass);
                        }

                        router.trigger('route:task', task.get('id'), 'view');
                    })
                    .fail((err, name, description) => {
                        Backbone.trigger('task:exec:finished');

                        if (self.ui.exec) {
                            self.ui.exec.removeClass(self.options.loadclass);
                        }

                        error.fromXHR({ xhr: err, name, description }, { autoClose: true });
                    });
            })
            .fail(e => {
                Backbone.trigger('task:exec:finished');

                self.showNotValidMsg(e);

                if (self.ui.exec) {
                    self.ui.exec.removeClass(self.options.loadclass);
                }
            });
    },

    onSave() {
        const self = this;

        this.ui.exec.addClass(this.options.loadclass);

        return this
            .validateAndSave()
            .done(() => {
                self.getTask().setAsDraft();
                app.addNotification({
                    message: 'Task saved',
                    options: {
                        autoClose: true,
                        reload: false
                    }
                });

                require('../../views/layouts/BasicLayout').showNotifications();
            })
            .fail(e => {
                self.showNotValidMsg(e);
            })
            .always(() => {
                self.ui.exec.removeClass(self.options.loadclass);
            });
    },

    /**
     * @description Validates all the custom fields of the task and tries to save it.
     * @returns {JQueryDeferred|Boolean} jqXHR .save() result or FALSE if custom fields validation failed
     */
    validateAndSave() {
        const task = this.getTask();
        const saved = $.Deferred(); // eslint-disable-line
        const generalFieldsErrors = task.validateGeneralFields();

        if (generalFieldsErrors.length === 0) {
            this.validateCustomFields()
                .done(() => {
                    task.save(null, { wait: false, parse: false })
                        .done(() => {
                            saved.resolve();
                            task.setAsDraft();
                        })
                        .fail(xhr => {
                            saved.reject();
                            error.fromXHR(xhr);
                        });
                })
                .fail(() => {
                    saved.reject();
                });
        } else {
            saved.reject();
        }

        return saved;
    },

    validateCustomFields() {
        const task = this.getTask();
        const customFieldsValidated = $.Deferred(); // eslint-disable-line
        const customFieldsCollection = this.customProps.currentView ?
            this.customProps.currentView.getCollection() :
            null;

        if (customFieldsCollection && customFieldsCollection.length > 0) {
            customFieldsCollection
                .validateFields(task)
                .done(() => {
                    customFieldsValidated.resolve();
                })
                .fail(errors => {
                    if (errors.results) {
                        errors.results.forEach(error => {
                            const fieldWithErr = customFieldsCollection.findWhere({ name: error.name });

                            if (fieldWithErr) {
                                fieldWithErr.trigger('invalid', { results: [error] });
                            }
                        });

                        const message = errors.results.length ?
                            'Some custom fields are not valid: <br/>' + errors.results.map(item => {
                                return (item.name ? (item.name + ': ') : '') + item.message;
                            }).join('<br/>') :
                            'Some custom fields are not valid';

                        error.message(message, { close: true });
                    } else {
                        error.message(errors.msg ? errors.msg : 'Error occured during validation request.');
                    }

                    customFieldsValidated.reject();
                });
        } else {
            customFieldsValidated.resolve();
        }

        return customFieldsValidated;
    },

    onModelChange() {
        app.setDraftTask(this.getTask());
    },

    onKillTimeChange(evtObj) {
        const time = evtObj.target.value;

        this.hideFieldValidateMessage('kill_timeout');

        this.getTask().set('kill_timeout', (time.includes(':') ? numeral().unformat(time) : time));
        this.getTask().validateKillTimeout();
    },

    onSpaceChange(evtObj) {
        const task = this.getTask();
        const requirements = task.get('requirements');

        this.hideFieldValidateMessage('disk_space');

        task.validateDiskSpace({ disk_space: evtObj.target.value });

        requirements.disk_space = numeral().unformat(evtObj.target.value);

        task.set('requirements', requirements);
    },

    onDescrChange() {
        const task = this.getTask();

        this.hideFieldValidateMessage('description');

        task.set('description', sanitize(this.ui.descr.val(), opts));
        task.validateDescription();
    },

    showFieldValidateMessage(fieldName, message) {
        this.ui[fieldName + 'PopupText'].text(message);
        this.ui[fieldName + 'Popup'].toggleClass('popup_visible', true);
    },

    hideFieldValidateMessage(fieldName) {
        this.ui[fieldName + 'Popup'].toggleClass('popup_visible', false);
    },

    getOwnerConfigView() {
        return new CreateTaskOwnerConfigView({
            groups: this.options.taskProps.groups,
            model: this.getTask(),
            user: this.options.user
        });
    },

    getCustomPropsView() {
        return new CreateTaskCustomFieldsView({
            model: this.getTask(),
            excludeContainer: true,
            taskProps: this.options.taskProps
        });
    },

    getPriorityConfigView() {
        return new PriorityConfig({
            model: this.getTask(),
            isScheduler: this.model instanceof SchedulerModel,
            groups: this.options.taskProps.groups
        });
    },

    getNotifyConfigView() {
        const task = this.getTask();

        return new CreateTaskNotifyConfig({
            model: task,
            taskProps: this.options.taskProps,
            collection: task.get('notifications')
        });
    },

    getAdvancedConfigView() {
        return new CreateTaskAdvancedView({
            model: this.getTask(),
            isScheduler: this.model instanceof SchedulerModel,
            taskProps: this.options.taskProps
        });
    },

    getSeTagConfigView() {
        return new CreateTaskSeTagConfigView({
            model: this.getTask()
        });
    },

    onValidationFail(err) {
        this.showFieldValidateMessage(err.field, err.message);
    },

    onClose() {
        require('../../views/layouts/BasicLayout').hideNotifications();

        this.stopListening(this.model, 'task:validating');

        const task = this.getTask();

        this.stopListening(task, 'change');
        this.stopListening(task, 'invalid:description');
        this.stopListening(task, 'invalid:kill_timeout');
        this.stopListening(task, 'invalid:disk_space');

        $(document).off('scroll', this.pinTaskControls);

        this.model.editMode = false;
    },

    showNotValidMsg(err) {
        const errorPopups = this.$el.find('.popup_visible');

        if (errorPopups.length) {
            $('body').animate({
                scrollTop: errorPopups.eq(0).offset().top - 80 // 80 to not be hovered with the top message
            }, 300);
        }

        if (!(app.error.currentView && app.error.currentView.$el)) {
            const reason = (err && (err.reason || (err.responseJSON && err.responseJSON.reason))) ||
                'Some fields are invalid';

            error.message(reason, {
                reload: false
            });
        }
    },

    initControlsPinning() {
        this.options.controlsDefaultOffset = this.ui.controlsCnt.offset().top;

        $(document).on('scroll', this.pinTaskControls);
        this.pinTaskControls();
    },

    pinTaskControls() {
        if (($(document).scrollTop() + $(window).innerHeight()) < this.options.controlsDefaultOffset) {
            if (this.ui.controlsCnt && this.ui.controlsCnt.addClass) {
                this.ui.controlsCnt.addClass('section__ctrl_fixed');
            }
        } else {
            if (this.ui.controlsCnt && this.ui.controlsCnt.addClass) { // eslint-disable-line
                this.ui.controlsCnt.removeClass('section__ctrl_fixed');
            }
        }
    },

    serializeData() {
        const task = this.getTask();

        return _.extend({}, task.toJSON(), {
            user: {
                login: this.options.user.get('login'),
                groups: this.options.user.getGroups().convertToList()
            }
        });
    }
});

module.exports = CreateTaskLayout;
