const ListModel = require('../../model/ListModel');
const CustomFieldView = require('./customProps/CustomFieldView');
const CreateTaskCustomListView = require('./customProps/CreateTaskCustomListView');
const CreateTaskCustomListSuggestView = require('./customProps/CreateTaskCustomListStaffView');

/**
 * @class CreateTaskListConfig
 * @extends Backbone.Marionette.ItemView
 */
module.exports = Marionette.Layout.extend({

    className: 'section__list section__list_ctrl',

    tagName: 'tr',

    template: require('./tpl/CreateTaskListConfig.hbs'),

    options: {
        isVisible: true,
        tipVisibleClass: 'popup_visible',
        defaultErrorMessage: 'Value is not valid'
    },

    ui: {
        addItem: '.add_item',
        popup: '.popup',
        popupOverlay: '.popup-overlay',
        errMsgCnt: '.popup__text'
    },

    events: {
        'click @ui.addItem': 'onAddItem'
    },

    modelEvents: {
        toggle: 'onToggle',
        invalid: 'showError',
        valid: 'hideError'
    },

    showError(errorObj) {
        const message = errorObj.results && errorObj.results[0].message ?
            errorObj.results[0].message :
            this.options.defaultErrorMessage;

        this.ui.errMsgCnt.text(message);
        this.ui.popup.toggleClass(this.options.tipVisibleClass, true);
    },

    hideError() {
        this.ui.popup.toggleClass(this.options.tipVisibleClass, false);
    },

    onToggle(toggle) {
        if (this.options.fieldViewMode === CustomFieldView.MODE.VIEW) {
            const self = this;

            if (!toggle) {
                setTimeout(() => {
                    self.close();
                }, 0);
            }
        } else {
            this.$el.toggle(toggle);
        }

        this.options.isVisible = toggle;
    },

    initialize(options) {
        this._generateCollection();

        this.model.collection.getByAttr = function (attr, value) {
            return this.detect(model => {
                /* eslint eqeqeq: 0 */
                return model.get(attr) == value;
            });
        };

        this.options = _.extend({}, this.options, options);
        this.setVisibility();

        if (this.options.fieldViewMode === CustomFieldView.MODE.VIEW) {
            if (this.model.get('modifiers').format === 'staff') {
                this.template = require('./customProps/tpl/view/StaffCustomListView.hbs');
            } else {
                this.template = require('./customProps/tpl/view/StringCustomListView.hbs');
            }
        }
    },

    setVisibility() {
        if (!this.model.isAvailable()) {
            this.options.isVisible = false;
        }

        return this.options.isVisible;
    },

    onRender() {
        this.addRegions({ dynamicList: '#dynamicList' });

        if (this.options.fieldViewMode !== CustomFieldView.MODE.VIEW) {
            this.showItems();
        }

        this.onToggle(this.options.isVisible);
        this.ui.popupOverlay.bemmyPopupOverlay();
    },

    onAddItem() {
        this._generateCollectionItem(null, null, 'add');
    },

    showItems() {
        const itemView = this.model.get('modifiers').format === 'staff' ?
            CreateTaskCustomListSuggestView :
            CreateTaskCustomListView;

        this.dynamicList.close();
        this.dynamicList.show(new Marionette.CollectionView({
            tagName: 'tbody',
            collection: this.collection,
            itemView,
            itemViewOptions: {
                fieldViewMode: this.options.fieldViewMode,
                taskProps: this.options.taskProps
            }
        }));
    },

    /**
     * Сохранение list / dict модели
     */
    onDynamicListChange() {
        let value = null;
        let name = this.name;
        let context = null;
        const resultList = new ListModel();

        resultList.urlRoot = this.url;

        this.collection.each(item => {
            /* eslint complexity: [1, 13] */

            const isDict = item.get('fieldType') === 'dict';

            context = context || item.get('context');
            name = name || item.get('name');

            if ((context && context.values) || isDict) {
                value = value || {};

                if (item.get('value') || item.get('contextValue')) {
                    value[item.get('value') || ''] = item.get('contextValue') || '';
                }
            } else {
                value = value || [];

                value.push(item.get('value') || '');
            }
        });

        this.options.taskModel.setCustomFieldValue(this.collection.name, value);

        const model = this.model.collection.getByAttr('name', this.collection.name);

        model.set('value', value);
    },

    /**
     * Генерируем коллекцию моделей list / dict по данным из ручек
     * @private
     */
    _generateCollection() {
        const self = this;
        const items = self.model.get('value');
        const isEmptyArray = items && items instanceof Array && !items.length;
        const isEmptyObject = items &&
            items.toString &&
            items.toString() === '[object Object]' &&
            Object.keys(items) &&
            Object.keys(items).length === 0;
        const isEmpty = !items || isEmptyArray || isEmptyObject;

        if (isEmpty) {
            self._generateCollectionItem();
        } else {
            _.forEach(items, (contextValue, value) => {
                if (items instanceof Array) {
                    value = contextValue;
                    contextValue = null;
                }

                if (self.collection) {
                    self._generateCollectionItem(value, contextValue, 'add');
                } else {
                    self._generateCollectionItem(value, contextValue);
                }
            });
        }

        if (self.collection) {
            self.collection.name = self.model.get('name');
        }
    },

    /**
     * Добавление модели в коллекцию
     * @param key - ключ (данные input)
     * @param val - значение (данные из select)
     * @param type - если установлен, то добавляем в коллекцию
     * @private
     */
    _generateCollectionItem(key, val, type) {
        const modifiers = this.model.get('modifiers');

        let attributes = {
            name: this.model.get('name'),
            context: this.model.get('context'),
            type: this.model.get('type'),
            fieldType: modifiers && modifiers.repetition,
            modifiers
        };

        if (key) {
            attributes = Object.assign(attributes, { value: key });
        }

        if (val) {
            attributes = Object.assign(attributes, { contextValue: val });
        }

        if (type) {
            this.collection.add(new ListModel(attributes));
        } else {
            this.collection = new Backbone.Collection(new ListModel(attributes));
        }

        this.collection.url =
            this.collection.url ||
            (this.model.get('parentModel') && this.model.get('parentModel').url() + '/custom/fields');
        this.listenTo(this.collection, 'change remove', this.onDynamicListChange, this.collection);
    },

    onClose() {
        this.stopListening(this.collection, 'change remove');
    }
});
