const MaillistSuggestGroupModel = require('../../model/suggest/filter/MaillistSuggestGroupModel');
const OwnerSuggestView = require('../controlSection/filter/suggest/OwnerSuggestView');
// eslint-disable-next-line no-unused-vars
const UserGroupModel = require('../../model/resources/UserGroupModel');
const SuggestNavigationBehavior = require('../behaviors/suggest/SuggestNavigationBehavior');

const tagTemplate = require('./tpl/tagTemplate.hbs');

/**
 * @class NotificationRecipientSuggestView
 * @extends OwnerSuggestView
 *
 * @property {NotificationModel} model
 * @property {object}            options
 * @property {boolean}           options.opened
 * @property {string}            options.openedClass
 * @property {boolean}           options.fetchMailingLists
 * @property {boolean}           options.onlyGroups
 * @property {boolean}           focusLost
 */
const NotificationRecipientSuggestView = OwnerSuggestView.extend({

    template: require('./tpl/NotificationRecipientSuggestView.hbs'),
    className: 'tag tag_size_fixed',

    ui: {
        menu: '.menu',
        input: '.tag__input',
        placeholder: '.tag__placeholder',
        tagClose: '.tag__close'
    },

    options: {
        opened: false,
        openedClass: 'tag__menu_visible',
        placeholder: 'Recipient',
        defaultValue: '',
        tagsSelector: '.tag__i'
    },

    events: _.extend({}, OwnerSuggestView.prototype.events, {
        'blur @ui.input': 'onBlur',
        'keyup @ui.input': 'onKeyup',
        'click @ui.tagClose': 'onTagClose',
        click: 'onContainerClick'
    }),

    modelEvents: {
        'change:recipients': 'onRecipientsChange',
        'invalid:recipients': 'onInvalidRecipients'
    },

    initialize() {
        this.focusLost = false;

        OwnerSuggestView.prototype.initialize.apply(this, arguments);
    },

    toggleMenu() {
        this.ui.menu.toggleClass(this.options.openedClass, !this.options.opened);
        this.options.opened = !this.options.opened;
    },

    showMenu() {
        this.options.opened = true;
        this.ui.menu.addClass(this.options.openedClass);
    },

    hideMenu() {
        this.options.opened = false;
        this.ui.menu.removeClass(this.options.openedClass);
    },

    onRender() {
        const self = this;

        setTimeout(() => {
            self.setInputStyling();
        }, 0);
    },

    /**
     * @override
     */
    onInput() {
        OwnerSuggestView.prototype.onInput.apply(this, arguments);
        this.showMenu();
    },

    onBlur() {
        this.focusLost = true;

        setTimeout(() => {
            if (this.focusLost) {
                this.focusLost = false;
                this.applySuggestedValue();
            }
        }, 250);
    },

    applySuggestedValue() {
        /* eslint max-statements: [1, 15] */

        this.ui.suggestItems = this.$('.autocomplete_item');

        const suggestItems = this.ui.suggestItems;
        const input = this.ui.input;

        if (suggestItems.length && suggestItems.first() && input.val()) {
            const currentItem = suggestItems.first();

            const suggestNavigation = (new SuggestNavigationBehavior());
            const selectedValue = suggestNavigation.getSelectedItemValue(currentItem);
            const recipients = this.model.attributes.recipients;

            if (recipients && recipients.includes(selectedValue)) {
                if (input) {
                    input.val('');
                }
            } else {
                this.trigger('selected', {
                    value: selectedValue,
                    type: currentItem.get(0).dataset.suggestType
                });

                setTimeout(() => {
                    this.model.attributes.recipients = _.uniq(this.model.attributes.recipients);
                }, 250);
            }
        } else {
            if (input && input.val) { // eslint-disable-line
                input.val('');
            }
        }
    },

    applyPlainValues() {
        const self = this;
        const plainValues = this.ui.input.val();

        if (plainValues) {
            plainValues
                .replace(/ ?, ?/ig, ' ')
                .split(' ')
                .filter(value => {
                    return value.length;
                })
                .forEach(plainValue => {
                    self.setValue(plainValue);
                });
        }
    },

    onItemSelect(selectedData) {
        if (this.focusLost) {
            this.focusLost = false;
        }

        this.setValue(selectedData.value);
        this.setInputStyling();
    },

    onRecipientsChange() {
        const self = this;
        const recipients = this.model.get('recipients');

        this.$(this.options.tagsSelector).remove();

        recipients.forEach(recipient => {
            const tags = self.$(self.options.tagsSelector);
            const tagHtml = tagTemplate({
                value: recipient
            });

            if (tags.length) {
                tags.last().after(tagHtml);
            } else {
                self.$el.prepend(tagHtml);
            }
        });

        this.setInputStyling();
    },

    onTagClose(evtObj) {
        evtObj.preventDefault();
        evtObj.stopPropagation();

        const recipients = this.model.get('recipients');
        const value = evtObj.currentTarget.dataset.value;
        const index = recipients.indexOf(value);

        if (index !== -1) {
            recipients.splice(index, 1);
            this.setRecipients(recipients);
        }
    },

    onContainerClick(evtObj) {
        evtObj.preventDefault();
        this.ui.input.focus();
    },

    onKeyup(event) {
        const currentValueLength = this.ui.input.val().length;

        this.ui.placeholder.toggle(!currentValueLength);

        if (currentValueLength === 0 && event.keyCode === 8) {
            const recipients = this.model.get('recipients');

            if (recipients.length) {
                recipients.pop();
            }

            this.setRecipients(recipients);
        }
    },

    /**
     * @override
     */
    setValue(valueToSet) {
        const updated = this.model.get('recipients');

        updated.push(valueToSet);
        this.setRecipients(updated);
        this.ui.input.val(this.options.defaultValue).trigger('input');
        this.toggleMenu();
    },

    /**
     * @override
     */
    serializeData() {
        return _.extend({}, OwnerSuggestView.prototype.serializeData.apply(this, arguments), {
            tags: this.model.get('recipients')
        });
    },

    setRecipients(recipients) {
        this.model.set('recipients', recipients);
        this.model.trigger('change:recipients');
    },

    setSuggestGroups() {
        OwnerSuggestView.prototype.setSuggestGroups.apply(this, arguments);

        if (this.options.fetchMailingLists) {
            this.collection.add(new MaillistSuggestGroupModel());
        }
    },

    /**
     * @param {NotificationModel} ccModel,
     * @param {array} notValidRecipients
     */
    onInvalidRecipients(ccModel, notValidRecipients) {
        const tags = this.$(this.options.tagsSelector);

        tags.removeClass('tag__err');

        notValidRecipients.forEach(item => {
            tags.each((index, elt) => {
                const tag = $(elt);

                if ($.trim(tag.text()) === item.toValid) {
                    tag.addClass('tag__err');

                    return false;
                }
            });
        });
    },

    setInputStyling() {
        const props = this.getInputStyling();

        this.ui.input.width(props.width);
        this.ui.placeholder.css({
            top: props.top,
            left: props.left
        });
    },

    getInputStyling() {
        const total = this.$el.innerWidth();
        let tagsWidth = NotificationRecipientSuggestView.TAGS_OUTER_OFFSETS;
        let tagTopOffset = 0;
        let defaultTagTopOffset = 0;

        this.$(this.options.tagsSelector).each((index, elt) => {
            const tag = $(elt);
            const topOffset = tag.offset().top;

            if (index === 0) {
                defaultTagTopOffset = topOffset;
            }

            // If tags starts from new line
            if (topOffset !== tagTopOffset) {
                tagTopOffset = topOffset;
                tagsWidth = NotificationRecipientSuggestView.TAGS_OUTER_OFFSETS;
            }

            tagsWidth += (tag.outerWidth() + NotificationRecipientSuggestView.TAGS_PADDINGS);
        });

        // Magic numbers are paddings
        return {
            width: (total - tagsWidth - 2),
            left: (tagsWidth - NotificationRecipientSuggestView.TAGS_OUTER_OFFSETS + 4),
            top: (tagTopOffset - defaultTagTopOffset + 4)
        };
    }
}, {
    TAGS_OUTER_OFFSETS: 18,
    TAGS_PADDINGS: 5
});

module.exports = NotificationRecipientSuggestView;
