import { includes } from 'lodash';

// eslint-disable-next-line no-unused-vars
import app from '../../../app';
import CustomFieldModel from '../../../model/CustomFieldModel';

/**
 * @class CustomView
 * @extends Backbone.Marionette.Layout
 *
 * @property {CustomFieldModel} model
 * @property {Object}           options
 * @property {TaskModel}        options.taskModel - Binded task model where custom values will be set
 */
const CustomFieldView = Marionette.Layout.extend(/** @lends CustomFieldView.prototype */{

    tagName: 'tr', // Need to add data-name to $el
    template: require('./tpl/edit/CustomFieldView.hbs'),

    options() {
        return {
            taskModel: null,
            isVisible: true,
            tipVisibleClass: 'popup_visible',
            defaultErrorMessage: 'Value is not valid'
        };
    },

    ui: {
        popup: '.popup',
        errMsgCnt: '.popup__text'
    },

    modelEvents: {
        'change:value': 'setTaskValue',
        invalid: 'showError',
        valid: 'hideError',
        toggle: 'onToggle' // Will be generated when model on what current model depends on was changed
        // used for sub fields, to remove self values from the task model when them are
        // not necessary.
    },

    initialize(options) {
        this.validID = null;
        this.options = _.extend({}, this.options, options);

        this.setVisibility();
        this.setTaskValue(this.model.get('value'));
    },

    onRender() {
        this.onToggle(this.options.isVisible);
    },

    onToggle(toggle) {
        if (this.options.fieldViewMode === CustomFieldView.MODE.VIEW) {
            // In view mode fields are displayed as a table which rows are even-colored, so we need
            // to remove not hide rows to prevent duplication of the rows with the same background.

            const self = this;

            if (!toggle) {
                setTimeout(() => {
                    self.close();
                }, 0);
            }
        } else {
            this.$el.toggle(toggle);
        }

        this.options.isVisible = toggle;

        this.setTaskValue(toggle ? this.model.get('value') : null);
    },

    onValueChanged(value) {
        this.hideError();
        this.model.set('value', value);
    },

    hideError() {
        if (this.validID) {
            clearTimeout(this.validID);
        }

        this.ui.popup.toggleClass(this.options.tipVisibleClass, false);
    },

    showError(errorObj) {
        this.validID = setTimeout(() => {
            const message = errorObj.results && errorObj.results[0].message ?
                errorObj.results[0].message :
                this.options.defaultErrorMessage;

            this.ui.errMsgCnt.text(message);
            this.ui.popup.toggleClass(this.options.tipVisibleClass, true);
        }, 700);
    },

    setTaskValue(value) {
        let _value = value;

        if (value && value.attributes) {
            _value = value.get('value');
        }

        this.options.taskModel.setCustomFieldValue(this.model.get('name'), _value);
    },

    setVisibility() {
        if (!this.model.isAvailable()) {
            this.options.isVisible = false;
        }

        return this.options.isVisible;
    },

    serializeData() {
        const serialized = _.clone(this.model.attributes);

        if (serialized.context && serialized.context.values) {
            serialized.context.values = this.serializeValues();
        }

        delete serialized.template;
        delete serialized.subFields;
        delete serialized.buildHelper;
        delete serialized.validateHelper;

        return serialized;
    },

    serializeValues() {
        const context = this.model.get('context') || {};

        if (context && !context.values) {
            context.values = { values: [['-- DEFAULT VALUE | looks smth went wrong --', '']] };
        }

        return _.map(context.values, item => {
            return {
                title: item[0],
                value: item[1]
            };
        });
    }
}, {

    /**
     * @description Factory for custom field's views
     * @param {CustomFieldModel} fieldModel
     */
    getView(fieldModel) {
        /* eslint max-statements: [1, 38] */
        /* eslint complexity: [1, 30] */

        let viewConstructor = CustomFieldView;

        if (typeof fieldModel.get(CustomFieldModel.EDIT_HELPER_NAME) === 'function') {
            // Field that will can have any control inside that will be built with helper
            return require('./UniqueCustomFieldView');
        }

        switch (fieldModel.get('type')) {
            case 'block': {
                viewConstructor = require('./InfoCustomFieldView');
                break;
            }
            case 'container':
            case 'resource': {
                viewConstructor = require('./ResourceCustomFieldView');
                break;
            }
            case 'task': {
                viewConstructor = require('./taskCustomProp/TaskCustomFieldView');
                break;
            }
            case 'boolean': {
                viewConstructor = require('./BooleanCustomFieldView');
                break;
            }
            case 'multiselect': {
                viewConstructor = require('./MultiselectCustomFieldView');
                break;
            }
            case 'text':
            case 'float':
            case 'string':
            case 'integer':
            case 'arcadia_url': {
                const context = fieldModel.get('context');
                const modifiers = fieldModel.get('modifiers');
                const repetition = modifiers && modifiers.repetition;
                const isList = repetition === 'list';
                const isDict = repetition === 'dict';

                if (repetition) {
                    const result = _.find([
                        { condition: isList, value: require('./../CreateTaskListConfig') },
                        { condition: isDict, value: require('./../CreateTaskListConfig') }
                    ], 'condition');

                    viewConstructor = result && result.value;

                    break;
                }

                if (context && _.isArray(context.values)) {
                    if (context.suggest || context.values.length > 7) {
                        viewConstructor = require('./SuggestCustomFieldView');
                    } else {
                        viewConstructor = require('./SelectCustomFieldView');
                    }
                } else if (
                    modifiers &&
                    modifiers.format &&
                    includes(['json', 'yaml'], modifiers.format.toLowerCase())
                ) {
                    viewConstructor = require('./FormattedCustomFieldView');
                } else if (modifiers && modifiers.multiline) {
                    viewConstructor = require('./TextCustomFieldView');
                } else if (modifiers && modifiers.format && modifiers.format.toLowerCase() === 'staff') {
                    viewConstructor = require('./StaffCustomFieldView');
                } else {
                    viewConstructor = require('./StringCustomFieldView');
                }

                break;
            }
            case 'select': {
                viewConstructor = require('./SelectCustomFieldView');
                break;
            }
            case 'radio': {
                viewConstructor = require('./RadioCustomFieldView');
                break;
            }
            default:
        }

        return viewConstructor;
    },

    MODE: {
        VIEW: 'view',
        EDIT: 'edit'
    }
});

module.exports = CustomFieldView;
