/* eslint camelcase: 0 */
const error = require('../../../error');
const CustomFieldView = require('./CustomFieldView');
const TaskResourceModel = require('../../../model/TaskResourceModel');
const ClipboardClientMixin = require('../../components/clipboardClient/ClipboardClientMixin');
const SelectResourceModalView = require('./SelectResourceModalView');
const TaskResourcesCollection = require('../../../collection/TaskResourcesCollection');

/**
 * @class ResourceCustomFieldView
 * @extends CustomFieldView
 *
 * @property {CustomFieldModel} model
 */
const ResourceCustomFieldView = CustomFieldView.mixin(ClipboardClientMixin).extend({

    template: require('./tpl/edit/ResourceCustomFieldView.hbs'),

    ui: _.extend({}, CustomFieldView.prototype.ui, {
        select: '.select_resource',
        remove: '.remove_resource',
        loader: '.task_resource_loader',
        popup: '.popup',
        popupOverlay: '.popup-overlay',

        resourceIdInput: '.resource_id_input input',
        replaceResourceTrigger: '.replace_resource'
    }),

    events: {
        'click @ui.remove': 'onRemove',
        'click @ui.select': 'onSelect',
        'click @ui.replaceResourceTrigger': 'onDirectIdChange',
        'paste @ui.resourceIdInput': 'onPaste'
    },

    modelEvents: _.extend({}, CustomFieldView.prototype.modelEvents, {
        'change:value': 'setRequirements',
        'change:resource_description': 'render',
        'change:resource_links': 'render'
    }),

    initialize() {
        CustomFieldView.prototype.initialize.apply(this, arguments);

        const context = (this.model.get('context'));

        this.options.isMultiple = Boolean(context && context.multiple);

        if (this.options.fieldViewMode === CustomFieldView.MODE.VIEW) {
            this.template = require('./tpl/view/ResourceCustomFieldView.hbs');
        }
    },

    setRequirements(model, nextValue) {
        if (model.get('title') === 'Tasks archive resource') {
            this.model
                .get('parentModel')
                .set({
                    requirements: Object.assign(
                        {},
                        this.model.get('parentModel').get('requirements'),
                        { tasks_resource: nextValue }
                    )
                });
        }

        this.render();
    },

    onRender() {
        this.initializeClipboardClient(
            '.resource-id-copier',
            'Resource ID has been copied to clipboard.'
        );

        this.initializeClipboardClient(
            '.skynet-proxy-copier',
            'Skynet copier link has been copied to clipboard.'
        );

        this.ui.popupOverlay.bemmyPopupOverlay();

        CustomFieldView.prototype.onRender.apply(this, arguments);
    },

    onPaste(evtObj) {
        const id = evtObj.originalEvent.clipboardData.getData('text/plain');

        if (id && /^\d+$/.test(id)) {
            this.ui.resourceIdInput.val(id);
            this.onDirectIdChange();
        }
    },

    onRemove(evtObj) {
        evtObj.preventDefault();

        const values = String(this.model.get('value')).split(',');

        if ((Array.isArray(values) ? values : [values]).includes(evtObj.currentTarget.dataset.resourceId)) {
            values.splice(values.indexOf(evtObj.currentTarget.dataset.resourceId), 1);
        }

        this.onValueChanged(values.join(','));
        this.render();
    },

    onSelect(evtObj) {
        evtObj.preventDefault();

        const self = this;
        const resources = this.getResourceCollection();

        this.ui.loader.removeClass('hidden');

        resources.update().done(() => {
            (new SelectResourceModalView({
                model: self.model,
                collection: resources,
                taskProps: self.options.taskProps
            })).show();
        }).always(() => {
            self.ui.loader.addClass('hidden');
        }).fail(() => {
            error.message('Error during loading resources');
        });
    },

    onDirectIdChange() {
        this.ui.replaceResourceTrigger.addClass('button_load');

        const self = this;
        const selectedId = this.ui.resourceIdInput.val();
        const resourceToReplace = new TaskResourceModel({ id: selectedId });

        resourceToReplace
            .fetch({ reset: true })
            .done(() => {
                const descriptionToSet = {};
                const linksToSet = {};

                descriptionToSet[resourceToReplace.get('id')] = resourceToReplace.get('description');
                linksToSet[resourceToReplace.get('id')] = resourceToReplace.get('http');

                self.model.set({
                    value: resourceToReplace.get('id'),
                    resource_description: descriptionToSet,
                    resource_links: linksToSet
                });
            })
            .fail(() => {
                error.message('Problem with fetching resource #' + selectedId + '. Please check if ID is right', {
                    reload: false
                });
            })
            .always(() => {
                self.ui.replaceResourceTrigger.removeClass('button_load');
            });
    },

    mapContextToFilter() {
        const context = this.model.get('context');
        const filterContextField = {};

        if (context) {
            Object.keys(context).forEach(key => {
                filterContextField[(key === 'types' ? 'type' : key)] = _.isArray(context[key]) ?
                    context[key].join(',') :
                    context[key];
            });
        }

        return filterContextField;
    },

    getResourceCollection() {
        return new TaskResourcesCollection([], _.extend(
            {
                pageCapacity: 25,
                state: 'READY'
            },
            this.mapContextToFilter()
        ));
    },

    serializeData() {
        const serialized = CustomFieldView.prototype.serializeData.apply(this, arguments);
        const ids = ((serialized.value === '' || _.isNull(serialized.value)) ?
            [] :
            String(serialized.value).split(','));

        serialized.isMultiple = this.options.isMultiple;
        serialized.resources = _.map(ids, id => {
            const url = serialized.resource_links && serialized.resource_links[id] ?
                serialized.resource_links[id].proxy :
                '';

            return {
                id,
                descr: (serialized.resource_description ? serialized.resource_description[id] : ''),
                url,
                skynet_id: (serialized.resource_skynet ? serialized.resource_skynet[id] : '')
            };
        });

        return serialized;
    }
});

module.exports = ResourceCustomFieldView;
