/* eslint max-len: [1, 122] */
import React from 'react';
import ReactDOM from 'react-dom';

import app from '../../../app';
import AutocompleteView from '../../controlSection/filter/selects/autocomplete/AutocompleteView';
import TaskResourceModel from '../../../model/TaskResourceModel';
import OwnerSuggestView from '../../controlSection/filter/suggest/OwnerSuggestView';
import FormAcceptBehavior from '../../behaviors/forms/FormAcceptBehavior';
import TaskTypeAutocompleteView from '../../controlSection/filter/selects/autocomplete/TaskTypeAutocompleteView';
import FilterPresetSavePromptView from '../../controlSection/filter/presetSaveConfirm/FilterPresetSavePromptView';
import DateTimeIntervalView from '../../controlSection/filter/selects/datetime/DateTimeIntervalView';

import ParameterInput from '../../../components/ParameterInput';
import { SuggestTagMulti } from '../../../components/SuggestTag';
import i18n from '../../../components/i18n';

/**
 * @class SelectResourceFilterView
 * @extends Backbone.Marionette.Layout
 *
 * @property {ResourceFilterModel} model
 */
const SelectResourceFilterView = Marionette.Layout.extend({

    template: require('./tpl/edit/SelectResourceFilter.hbs'),

    className() {
        return (this.options.isModal ? 'modal__filter' : '');
    },

    options: {
        isModal: true
    },

    ui: {
        arch: '.res_arch select',
        status: '.res_status select',
        taskId: '.task_id',
        taskIdInput: '.task_id input',
        selfId: '.resource_id',
        selfIdInput: '.resource_id input',
        resourceId: '.depend_id',
        resourceIdInput: '.depend_id input',
        attrs: '.attr_attrs',
        tag: '.attr_tags',
        hidden: '.hidden',

        apply: '.resource_filter_apply',
        reset: '.resource_filter_reset',
        save: '.resource_filter_save'
    },

    events: {
        'click @ui.apply': 'onApply',
        'click @ui.reset': 'onReset',
        'click @ui.save': 'onFilterSave',

        'change @ui.arch': 'onArchChange',
        'change @ui.status': 'onStatusChange',
        'bemmy:set @ui.hidden': 'onHiddenChange',
        'bemmy:refresh @ui.taskId': 'onTaskIdChange',
        'bemmy:refresh @ui.resourceId': 'onDependIdChange',
        'bemmy:refresh @ui.selfId': 'onSelfIdChange'
    },

    modelEvents: {
        blank: 'render',
        'change:attrs': 'renderBlankControls',
        'change:any_params': 'renderBlankControls',
        'change:tags': 'renderBlankControls',
        'change:all_tags': 'renderBlankControls',
        'change:created': 'showCreatedRegion'
    },

    behaviors: {
        FormAcceptBehavior: {}
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);
        this.listenTo(this, FormAcceptBehavior.EVENTS.ACCEPT_EVENT, this.onFormAccept);

        this.renderBlankControls = this.renderBlankControls.bind(this);
        this.renderControls = this.renderControls.bind(this);
        this.handleTagsSet = this.handleTagsSet.bind(this);
        this.handleTagsAllSet = this.handleTagsAllSet.bind(this);
        this.handleAttributesChange = this.handleAttributesChange.bind(this);
        this.handleAttributesToggleChange = this.handleAttributesToggleChange.bind(this);
    },

    onRender() {
        this.$('.input').bemmyInput();
        this.$('.check').bemmyCheck();
        this.$('.select').bemmySelect();
        this.$('.popup-overlay').bemmyPopupOverlay();

        this.ownerSuggest.close();
        this.typeSelector.close();
        this.hostSelector.close();

        this.ownerSuggest.show(this.getOwnerSuggest());
        this.typeSelector.show(this.getTypeSelector());
        this.hostSelector.show(this.getHostSelector());
        this.showCreatedRegion();

        this.renderBlankControls();
    },

    renderBlankControls() {
        this.renderControls();
    },

    renderControls(model, tags) {
        const _tags = this.model.get('tags');
        const togglerValue = Boolean(this.model.get('all_tags'));
        const attrs = this.model.get('attrs');
        const anyParams = this.model.get('any_params') === null ? false : this.model.get('any_params');

        if (this.ui.attrs[0]) {
            ReactDOM.render(
                <ParameterInput
                    value={attrs}
                    placeholder="Attributes"
                    togglerValue={anyParams}
                    onChange={this.handleAttributesChange}
                    onToggle={this.handleAttributesToggleChange}
                    hasExtendedTypes
                    />,
                this.ui.attrs[0]
            );
        }

        if (this.ui.tag[0]) {
            ReactDOM.render(
                <SuggestTagMulti
                    value={tags || (_tags instanceof Array ? _tags : _tags.split(','))}
                    togglerValue={togglerValue}
                    name={'tags'}
                    placeholder={i18n.labels.tags}
                    onChange={this.handleTagsSet}
                    onToggle={this.handleTagsAllSet}
                    />,
                this.ui.tag[0]
            );
        }
    },

    handleAttributesChange(value) {
        this.model.set('attrs', value);
    },

    handleAttributesToggleChange(value) {
        this.model.set('any_params', value);
    },

    handleTagsSet(tags) {
        this.model.set('tags', tags);
    },

    handleTagsAllSet(flag) {
        this.model.set('all_tags', flag);
    },

    getTypeSelector() {
        return new TaskTypeAutocompleteView({
            lazy: 500,
            items: this.getTypeSuggestItems(),
            value: this.model.get('type'),
            model: this.model,
            showAnyItem: true,
            placeHolder: 'Types',
            modelTargetKey: 'type',
            selectClassName: 'resource_filter_type'
        });
    },

    showCreatedRegion() {
        this.createInterval.show(this.getCreateIntervalRegion());
    },

    getCreateIntervalRegion() {
        return new DateTimeIntervalView({
            model: this.model.get('created'),
            dateOnly: false,
            presets: [
                { key: '3_days', label: 'Last 3 days' },
                { key: '7_days', label: 'Last 1 week' },
                { key: '14_days', label: 'Last 2 weeks' },
                { key: '1_months', label: 'Last month' }
            ],
            placeholder: 'Created',
            classNamePrefix: 'app_filter_create'
        });
    },

    getTypeSuggestItems() {
        const types = String(this.model.get('type')).split(',');

        if (types.length > 1) {
            return new Backbone.Collection(this.options.taskProps.resourceTypes.filter(type => {
                for (let q = 0; q < types.length; q++) {
                    if (types[q] === type.get('type')) {
                        return true;
                    }
                }

                return false;
            }));
        }

        return this.options.taskProps.resourceTypes;
    },

    getOwnerSuggest() {
        return new OwnerSuggestView({
            model: this.model,
            ownerGroups: this.options.taskProps.groups.convertToList(),
            defaultValue: this.model.get('owner'),
            modelTargetKey: 'owner'
        });
    },

    getHostSelector() {
        return new AutocompleteView({
            lazy: 200,
            items: this.options.taskProps.hosts,
            value: this.model.get('client'),
            model: this.model,
            showAnyItem: true,
            placeHolder: 'Client',
            modelTargetKey: 'client',
            selectClassName: 'app_filter_client'
        });
    },

    serializeData() {
        return _.extend(
            { isModal: this.isModal },
            this.model.toJSON(),
            this.options.taskProps,
            { statuses: TaskResourceModel.getPossibleStatuses() }
        );
    },

    onHiddenChange(evtObj, value) {
        this.model.set('hidden', value);
    },

    onArchChange(evtObj) {
        this.model.set('arch', evtObj.target.value);
    },

    onStatusChange(evtObj) {
        this.model.set('state', evtObj.target.value);
    },

    onTaskIdChange() {
        this.model.set('task_id', this.ui.taskIdInput.val());
    },

    onDependIdChange() {
        this.model.set('depend_by', this.ui.resourceIdInput.val());
    },

    onSelfIdChange() {
        this.model.set('id', this.ui.selfIdInput.val());
    },

    onModelChange() {
        this.ui.apply.addClass('button_type_hl');
    },

    onApply() {
        const filterValues = this.model.clone();

        this.ui.apply.removeClass('button_type_hl');

        app.trigger('change:filterparams:preset', filterValues);

        // Filter state should be saved only for non-modal grids. (modal grids are in almost for custom parameters)
        if (!this.options.isModal) {
            this.updateLastFilter(filterValues);
        }
    },

    onReset(evtObj) {
        evtObj.preventDefault();
        this.model.resetFilter();

        return false;
    },

    onFilterSave(evtObj) {
        evtObj.preventDefault();

        (new FilterPresetSavePromptView({
            model: this.model.clone(),
            collection: this.options.user.getPresets()
        })).show();
    },

    onFormAccept() {
        this.onApply();
    },

    updateLastFilter(filterValues) {
        const defaultFilter = app.request('DEFAULT_FILTER');

        defaultFilter.resourceList.set(filterValues);
        defaultFilter.resourceList.saveFilterState();
    },

    onClose() {
        this.stopListening(this, FormAcceptBehavior.EVENTS.ACCEPT_EVENT);
    }
});

module.exports = SelectResourceFilterView;
