const TaskModel = require('../../../model/TaskModel');

/**
 * @class PriorityConfig
 * @extends Marionette.ItemView
 *
 * @property {TaskModel} model
 */
const PriorityConfig = Marionette.ItemView.extend({

    className: 'b b b_gap-x_M',

    template: require('./tpl/PriorityConfig.hbs'),

    ui: {
        priorityClass: '.priority-class',
        prioritySubClass: '.priority-subclass',
        priorityClassBtns: '.priority-class .button',
        prioritySubClassBtns: '.priority-subclass .button'
    },

    events: {
        'bemmy:set @ui.priorityClass': 'onPriorityClassChange',
        'bemmy:set @ui.prioritySubClass': 'onPrioritySubClassChange'
    },

    modelEvents: {
        'change:owner': 'onOwnerChange'
    },

    initialize(options) {
        this.options = {
            groups: options.groups,
            isScheduler: options.isScheduler,
            disableClassName: 'button_disable button_disable_events'
        };
    },

    onRender() {
        this.$('.button').bemmyButton();

        this.refreshEnabledPriorityLevel();
    },

    onOwnerChange() {
        this.refreshEnabledPriorityLevel();
    },

    onPriorityClassChange(evtObj, priority) {
        evtObj.preventDefault();

        this.model.setPriorityClass(priority);
        this.model.trigger('change');

        this.refreshEnabledPriorityLevel();
    },

    onPrioritySubClassChange(evtObj, priority) {
        evtObj.preventDefault();
        this.model.setPrioritySubClass(priority);
        this.model.trigger('change');
    },

    refreshEnabledPriorityLevel() {
        /* eslint max-statements: [1, 27] */
        /* eslint complexity: [1, 13] */

        const owner = this.model.get('owner');

        if (!this.options.isScheduler) {
            this.ui.priorityClassBtns.removeClass(this.options.disableClassName);
            this.ui.prioritySubClassBtns.removeClass(this.options.disableClassName);
        }

        if (owner && owner.toLowerCase() !== owner) { // Groups
            const group = this.options.groups.findWhere({ name: owner });

            if (!group) {
                return;
            }

            const priority = this.model.get('priority');
            const availablePriority = group.get('priority_limits').ui;

            if (TaskModel.PRIORITY.CLASS.indexOf(priority.class) >
                TaskModel.PRIORITY.CLASS.indexOf(availablePriority.class)) {
                this.model.setPriorityClass(TaskModel.PRIORITY.CLASS[0]);
                this.model.setPrioritySubClass(TaskModel.PRIORITY.SUBCLASS[0]);

                this.render();

                return;
            }

            if (priority.class === availablePriority.class) {
                if (TaskModel.PRIORITY.SUBCLASS.indexOf(priority.subclass) >
                    TaskModel.PRIORITY.SUBCLASS.indexOf(availablePriority.subclass)) {
                    this.model.setPrioritySubClass(TaskModel.PRIORITY.SUBCLASS[0]);

                    this.render();

                    return;
                }
            }

            if (availablePriority.class === TaskModel.PRIORITY.CLASS[0]) { // BACKGROUND
                this.ui.priorityClassBtns.slice(1).addClass(this.options.disableClassName);
            } else if (availablePriority.class === TaskModel.PRIORITY.CLASS[1]) { // SERVICE
                this.ui.priorityClassBtns.slice(2).addClass(this.options.disableClassName);
            }

            if (priority.class === availablePriority.class) {
                if (availablePriority.subclass === TaskModel.PRIORITY.SUBCLASS[0]) { // LOW
                    this.ui.prioritySubClassBtns.slice(1).addClass(this.options.disableClassName);
                } else if (availablePriority.subclass === TaskModel.PRIORITY.SUBCLASS[1]) { // MEDIUM
                    this.ui.prioritySubClassBtns.slice(2).addClass(this.options.disableClassName);
                }
            }
        }
    },

    serializeData() {
        const priority = this.model.get('priority');

        return {
            priority: {
                mainclass: [
                    {
                        label: 'Background',
                        value: TaskModel.PRIORITY.CLASS[0],
                        active: (priority && priority.class === TaskModel.PRIORITY.CLASS[0])
                    },
                    {
                        label: 'Service',
                        value: TaskModel.PRIORITY.CLASS[1],
                        active: (priority && priority.class === TaskModel.PRIORITY.CLASS[1])
                    },
                    {
                        label: 'User',
                        value: TaskModel.PRIORITY.CLASS[2],
                        active: (priority && priority.class === TaskModel.PRIORITY.CLASS[2]),
                        disable: this.options.isScheduler
                    }
                ],
                subclass: [
                    {
                        label: 'Low',
                        value: TaskModel.PRIORITY.SUBCLASS[0],
                        active: (priority && priority.subclass === TaskModel.PRIORITY.SUBCLASS[0])
                    },
                    {
                        label: 'Normal',
                        value: TaskModel.PRIORITY.SUBCLASS[1],
                        active: (priority && priority.subclass === TaskModel.PRIORITY.SUBCLASS[1])
                    },
                    {
                        label: 'High',
                        value: TaskModel.PRIORITY.SUBCLASS[2],
                        active: (priority && priority.subclass === TaskModel.PRIORITY.SUBCLASS[2])
                    }
                ]
            }
        };
    }

});

module.exports = PriorityConfig;
