/* eslint max-statements: [1, 15] */

// eslint-disable-next-line no-unused-vars
const app = require('../../../app');
const ContentLayout = require('./ContentLayout');
const ClientsMainNavigationView = require('../../navigation/clients/ClientsMainNavigation');
const ClientsGroupsView = require('../../clients/ClientsGroupsView');
const ClientsGroupsCollection = require('../../../collection/ClientsGroupsCollection');
const ClientsGroupModel = require('../../../model/clients/ClientsGroupModel');
const ClientsSecondaryNavigation = require('../../navigation/clients/ClientsSecondaryNavigation');
const ClientsFilterLayout = require('./../../clients/ClientsFilterLayout');
const ClientsBatchControls = require('../../controlSection/clients/ClientsBatchControlsView');

/**
 * @class ClientsContentLayout
 * @extends ContentLayout
 *
 * @property {CoreApplication} model
 */
const ClientsContentLayout = ContentLayout.extend({

    className: 'section clients__list_compact',

    template: require('./tpl/ClientsContentLayout.hbs'),

    options: {
        sortGroups: {},
        activeGroup: 'dc'
    },

    ui: {
        loader: '.app_clients_loader'
    },

    collectionEvents: {
        filter: 'onFilterApply'
    },

    clientsGroups: null,
    droppingFilter: false,

    initialize(options) {
        this.options = _.extend({}, this.options, options);
        this.collection = new Backbone.VirtualCollection(this.model.getClients());
        this.clientsGroups = new ClientsGroupsCollection();
        this.options.sortGroups = this.model.getClients().getSortGroups();

        this.setClientsGroups(this.options.activeGroup);

        this.listenTo(this, 'selected:group', this.onGroupSelected);
        this.listenTo(this, 'change:mode', this.setViewMode);
        this.listenTo(this, 'show:dead', this.showDeadClients);
        this.listenTo(this.collection.collection, 'update:start', this.showLoader.bind(this));
        this.listenTo(this.collection.collection, 'update:complete', this.hideLoader.bind(this));

        if (this.model.getPageProps().isAutoupdateEnabled) {
            this.listenTo(Backbone, 'autoupdate:on', this.turnOnAutoupdate);
            this.listenTo(Backbone, 'autoupdate:off', this.turnOffAutoupdate);
        }

        this.listenTo(this.model.getClients(), 'sync', function () {
            this.options.sortGroups = this.model.getClients().getSortGroups();
            this.setClientsGroups(this.options.activeGroup);
        });
    },

    onRender() {
        if (this.navigation) {
            this.navigation.close();
            this.navigation.show(this.getNavigation());
        }

        this.secNavigation.close();
        this.filter.close();
        this.batchControls.close();
        this.content.close();

        this.secNavigation.show(this.getSecondaryNavigation());
        this.filter.show(this.getFilter());
        this.batchControls.show(this.getBatchControls());
        this.content.show(this.getContent());
    },

    onFilterApply() {
        if (!this.droppingFilter) {
            this.setClientsGroups(this.options.activeGroup);

            this.collection.trigger('batch:deselect');
        }
    },

    onGroupSelected(groupName) {
        // This.dropGroupsFilter();
        this.setClientsGroups(groupName);
        this.options.activeGroup = groupName;
    },

    showLoader() {
        this.ui.loader.fadeIn(150);
        this.ui.loader.removeClass('loader_hide');
    },

    hideLoader() {
        this.ui.loader.fadeOut(300);
        this.ui.loader.addClass('loader_hide');
    },

    dropGroupsFilter() {
        this.droppingFilter = true;
        this.collection.updateFilter(() => {
            return true;
        });
        this.droppingFilter = false;
    },

    getNavigation() {
        return new ClientsMainNavigationView({
            model: this.model,
            parent: this,
            sortGroups: this.options.sortGroups,
            activeGroup: this.options.activeGroup
        });
    },

    getSecondaryNavigation() {
        return new ClientsSecondaryNavigation({
            model: this.model,
            parent: this,
            sortGroups: this.options.sortGroups,
            collection: this.clientsGroups
        });
    },

    getFilter() {
        return new ClientsFilterLayout({
            model: this.model.getClients().getFilterParams().clone(),
            props: this.model.getProps(),
            collection: this.collection
        });
    },

    getBatchControls() {
        return new ClientsBatchControls({
            user: this.model.getUser(),
            collection: this.collection
        });
    },

    getContent() {
        return new ClientsGroupsView({
            user: this.model.getUser(),
            model: this.model,
            parent: this,
            collection: this.clientsGroups
        });
    },

    setClientsGroups(groupName) {
        if (groupName === 'tags') {
            return this.setTagsClientsGroups(groupName);
        }
        return this.setOrdinaryClientsGroups(groupName);
    },

    setOrdinaryClientsGroups(groupName) {
        const groups = [];

        if (this.options.sortGroups[groupName] && this.options.sortGroups[groupName].length > 0) {
            // Creating basic set of empty collections
            this.options.sortGroups[groupName].forEach(function (subGroup) {
                const emptyGroup = new ClientsGroupModel({ title: subGroup, parentGroup: groupName });

                emptyGroup._filterKey = this.prepareFilterKey(groupName);
                emptyGroup._filterValue = subGroup;

                groups.push(emptyGroup);
            }, this);

            // Setting values into the collections
            this.collection.forEach(function (client) {
                // Dropping selected flag
                client.attributes.selected = false;

                this.options.sortGroups[groupName].forEach((subGroup, index) => {
                    const clientFilterValue = ClientsGroupModel.getFilterValue(client, groups[index]._filterKey);

                    if ((Array.isArray(clientFilterValue) ?
                        clientFilterValue :
                        [clientFilterValue]).includes(groups[index]._filterValue)
                    ) {
                        // Not using .add to not trigger any events
                        groups[index].attributes.items.models.push(client);
                        groups[index].attributes.items._addReference(client);
                        groups[index].attributes.items.length++;
                    }
                }, this);
            }, this);
        }

        this.clientsGroups.set(_.sortBy(groups, group => {
            return group.get('items').length * -1;
        }));
    },

    setTagsClientsGroups(groupName) {
        const tags = Object.keys(this.options.sortGroups[groupName]);
        const groups = [];

        if (tags && tags.length > 0) {
            // Root types
            tags.forEach(function (subGroup) {
                const emptyGroup = new ClientsGroupModel({
                    title: subGroup,
                    parentGroup: groupName,
                    rootGroup: subGroup
                });

                emptyGroup._filterKey = this.prepareFilterKey(groupName);
                emptyGroup._filterValue = subGroup;

                groups.push(emptyGroup);
            }, this);

            // Setting values into the collections
            this.collection.forEach(function (client) {
                // Dropping selected flag
                client.attributes.selected = false;

                tags.forEach(function (subGroup, index) {
                    this.options.sortGroups[groupName][subGroup].forEach(item => {
                        const clientFilterValue = ClientsGroupModel.getFilterValue(client, groups[index]._filterKey);

                        if ((Array.isArray(clientFilterValue) ?
                            clientFilterValue :
                            [clientFilterValue]).includes(item)) {
                            groups[index].attributes.subs = groups[index].attributes.subs || {};
                            groups[index].attributes.subs[item] = groups[index].attributes.subs[item] || [];
                            groups[index].attributes.subs[item].push(client);

                            groups[index].attributes.items.models.push(client);
                            groups[index].attributes.items._addReference(client);
                            groups[index].attributes.items.length++;
                        }
                    });
                }, this);
            }, this);
        }

        this.clientsGroups.set(_.sortBy(groups, group => {
            return group.get('items').length * -1;
        }));
    },

    prepareFilterKey(title) {
        switch (title) {
            case 'os': {
                return function (client) {
                    const platforms = client.get('platforms');

                    return (_.isNull(platforms) || !platforms.length ? client.get('platform') : platforms);
                };
            }
            case 'ram': {
                return function (client) {
                    return ('~' + Math.floor(client.get('ram') / 1024 / 1024 / 1024) + ' GB');
                };
            }
            case 'type': {
                return function (client) {
                    return client.get('task').task_type;
                };
            }
            case 'owner': {
                return function (client) {
                    return client.get('task').owner;
                };
            }
            default: return title;
        }
    },

    setViewMode(mode) {
        this.$el.toggleClass('clients__list_compact', (mode === 'compact'));
        this.clientsGroups.trigger('change:mode', mode);
    },

    turnOnAutoupdate(interval) {
        this.model.getClients().activateAutoupdate(interval);
    },

    turnOffAutoupdate() {
        this.model.getClients().deActivateAutoupdate();
    },

    showDeadClients() {
        this.collection.updateFilter({ alive: false });
    },

    onClose() {
        this.turnOffAutoupdate();

        this.stopListening(this.model.getClients());
        this.stopListening(Backbone, 'autoupdate:on');
        this.stopListening(Backbone, 'autoupdate:off');

        this.stopListening(this, 'selected:group');
        this.stopListening(this, 'change:mode');
        this.stopListening(this, 'show:dead');
        this.stopListening(this.collection.collection, 'update:start');
        this.stopListening(this.collection.collection, 'update:complete');
    }
});

module.exports = ClientsContentLayout;
