/* eslint max-statements: [1, 19] */
const React = require('react');
const ReactDOM = require('react-dom');
const Provider = require('react-redux').Provider;
const store = require('../../../store/initializedStore');

const app = require('../../../app');
const error = require('../../../error');
// eslint-disable-next-line no-unused-vars
const router = require('../../../router');
const ContentLayout = require('./ContentLayout');
const TaskModel = require('../../../model/TaskModel');
const BasicCollection = require('../../../collection/BasicCollection');
const SingleTaskControlSectionLayout = require('../../controlSection/singleTask/SingleTaskControlSectionLayout');
const SingleTaskCommonViewFactory = require('../../singleTask/SingleTaskCommonViewFactory');
const SingleTaskResourcesView = require('../../singleTask/SingleTaskResourcesView');
const SingleTaskGraphicsView = require('../../singleTask/SingleTaskGraphicsView');
const SingleTaskContextView = require('../../singleTask/SingleTaskContextView');
const SingleTaskHistoryView = require('../../singleTask/SingleTaskHistoryView');
const SingleTaskHostsView = require('../../singleTask/SingleTaskHostsView');
const SingleTaskSubsView = require('../../singleTask/SingleTaskSubsView');
const AutoUpdateFaviconMixin = require('../../components/autoUpdateFavicon/AutoUpdateFaviconMixin');
const PushNotificationsMixin = require('../../components/pushNotifications/PushNotificationsMixin');

const LogsTab = require('../../../components/LogsTab');
const ReportTab = require('../../../components/ReportTab');

/**
 * @class SingleTaskContentLayout
 * @extends ContentLayout
 *
 * @property {CoreApplication} model
 * @property {Marionette.Region} content
 * @property {Marionette.Region} controls
 * @property {Marionette.Region} navigation
 */
const SingleTaskContentLayout = ContentLayout.mixin(AutoUpdateFaviconMixin).mixin(PushNotificationsMixin).extend({

    className: 'section section_slim',

    template: require('./tpl/SingleTaskCompactContentLayout.hbs'),

    ui: {
        loader: '.preloader'
    },

    options: {
        loaderMessage: 'Loading task data..',
        contentConfig: {
            view: {
                failable: true,
                fetchMethod: 'fetchCustomParameters',
                canBeUpdated: true, // False
                collectionProp: null,
                contentViewClass: SingleTaskCommonViewFactory
            },
            logs: {
                fetchMethod: false,
                canBeUpdated: true,
                collectionProp: null,
                contentReactViewClass: LogsTab
            },
            hosts: {
                fetchMethod: 'fetchHosts',
                canBeUpdated: true,
                collectionProp: 'hosts',
                contentViewClass: SingleTaskHostsView
            },
            depend: {
                fetchMethod: 'fetchDependOn',
                canBeUpdated: true,
                collectionProp: 'requirements.resources',
                contentViewClass: SingleTaskResourcesView
            },
            context: {
                fetchMethod: 'fetchContext',
                canBeUpdated: true,
                collectionProp: null,
                contentViewClass: SingleTaskContextView
            },
            history: {
                fetchMethod: 'fetchAudit',
                canBeUpdated: true,
                collectionProp: null,
                contentViewClass: SingleTaskHistoryView
            },
            children: {
                fetchMethod: 'fetchChildren',
                canBeUpdated: true,
                collectionProp: 'children',
                selectableGrid: true,
                contentViewClass: SingleTaskSubsView
            },
            dependant: {
                fetchMethod: 'fetchDependant',
                canBeUpdated: true,
                collectionProp: 'dependant',
                selectableGrid: true,
                contentViewClass: SingleTaskSubsView
            },
            resources: {
                fetchMethod: 'fetchResources',
                canBeUpdated: true,
                collectionProp: 'resources',
                contentViewClass: SingleTaskResourcesView
            },
            hardware: {
                fetchMethod: 'fetchRelatedHosts',
                canBeUpdated: false,
                collectionProp: 'hardware',
                contentViewClass: SingleTaskGraphicsView
            },
            report: {
                fetchMethod: false,
                canBeUpdated: false,
                collectionProp: null,
                contentReactViewClass: ReportTab
            }
        }
    },

    modelEvents: {
        'change:pageProps'() {
            this.trigger('content:replaced');
        }
    },

    /** REGIONS DEFINED DIRECTLY IN TEMPLATE */
    regions: {
        content: '#single-task-content'
    },

    initialize(options) {
        this.autoUpdateIntervalID = null;
        this.initializeAutoupdateFaviconClient();
        this.initializePushNotificationsClient();

        this.options = _.extend({}, this.options, options);

        this._model = this.getModel();

        this.listenTo(this, 'content:replace', this.onChangeContent);

        if (this.model.getPageProps().isAutoupdateEnabled) {
            this.deActivateEvents();
            this.activateEvents();
        }
    },

    onRender() {
        document.title = ('Task #' + this._model.get('id') + ' ' + this._model.get('type'));

        this.showContent(true);
    },

    showContent(withControls, fetchOnly) {
        const self = this;
        const task = this.getModel();

        if (task && task.get('status') === TaskModel.STATUS.RELEASED) {
            task.fetchReleaseInfo()
                .done(() => {
                    if (withControls) {
                        self.controls.close();
                        self.controls.show(self.getControls());
                    }

                    self.setContent(self.getAction(), fetchOnly);
                })
                .fail(xhr => {
                    app.reqres.request('ERR_TRACE', xhr);
                    error.showErrPage(xhr);
                });
        } else {
            if (withControls) {
                self.controls.close();
                self.controls.show(self.getControls());
            }

            self.setContent(self.getAction(), fetchOnly);
        }
    },

    onChangeContent(options) {
        this.setContent(options.action);
    },

    setContent(action, fetchOnly) {
        const parentAction = action.split('-')[0];

        const self = this;
        const pageProps = this.model.getPageProps();
        const isUpdateDisabled = fetchOnly &&
            (pageProps.forceContentAutoupdateDisabledForAction || []).includes(parentAction);

        this.showLoader();

        if (this.options.contentConfig[parentAction] && !isUpdateDisabled) {
            const failable = this.options.contentConfig[parentAction].failable;
            const fetchMethod = this.options.contentConfig[parentAction].fetchMethod;

            jQuery
                .when(_.isFunction(this.getModel()[fetchMethod]) ? this.getModel()[fetchMethod]() : true)
                .always((xhr, status, response) => {
                    // On success and error will have different order of arguments
                    // Docs link: http://api.jquery.com/jQuery.ajax/#jqXHR
                    xhr = (xhr.status && xhr.responseText ? xhr : response);

                    if (xhr && (xhr.status < 200 || xhr.status >= 300)) {
                        if (failable) {
                            error.fromXHR(xhr, { autoClose: false });
                            self.processFetchedContent(action, fetchOnly);
                        } else {
                            error.fromXHR(xhr);
                            self.trigger('content:replaced');
                        }
                    } else { // Truly resolve of
                        self.processFetchedContent(action, fetchOnly);
                    }
                });
        }
    },

    processFetchedContent(action, fetchOnly) {
        if (fetchOnly && this.content && action !== 'children') {
            const view = this.content.currentView;

            if (view && view.collection) {
                const collection = this.getContentDataCollection(this.options.contentConfig[action].collectionProp);

                if (collection) {
                    view.collection.set(collection.models);
                }
            }
        } else {
            this.applyContent(action);
        }
    },

    applyContent(action) {
        const updatedPageProps = _.extend({}, this.model.getPageProps(), { action });
        const ReactView = this.actionReactView(action);

        this.hideLoader();

        if (ReactView) {
            if (this.content) {
                if (!this.content.$el) {
                    this.content.ensureEl();
                }
                ReactDOM.render(<Provider store={store}>
                    <ReactView model={this.model.get('task')} action={action}/>
                </Provider>, this.content.$el[0]);
            }
        } else {
            this.destroyReactView();

            if (this.content) {
                this.content.close();
                this.content.show(this.getContent(action));
            }
        }

        this.model.setPageProps(updatedPageProps);
    },

    getModel() {
        return this.model.getTask();
    },

    getContent(action) {
        const contentConfig = this.options.contentConfig[action];

        if (contentConfig && contentConfig.contentViewClass) {
            const ContentView = contentConfig.contentViewClass;
            const collectionProp = contentConfig.collectionProp;
            const contentInstanceOptions = {
                app: this.model,
                model: this.getModel(),
                collection: this.getContentDataCollection(collectionProp)
            };

            if (contentConfig.contentViewClass.factory) {
                return contentConfig.contentViewClass.factory(
                    contentInstanceOptions,
                    contentConfig
                );
            }

            return new ContentView(contentInstanceOptions);
        }
        /* eslint no-console: 0 */
        console.error('SANDBOX: Content for «' + action + '» action is not specified.');
    },

    getContentDataCollection(collectionProp) {
        if (!collectionProp) {
            return null;
        }

        if (!collectionProp.includes('.')) {
            return this.getModel().get(collectionProp);
        }
        collectionProp = collectionProp.split('.');

        return this.getModel().get(collectionProp[0])[collectionProp[1]];
    },

    getControls() {
        return new SingleTaskControlSectionLayout({
            layout: this,
            coreApp: this.model,
            taskProps: this.model.getProps()
        });
    },

    getAction() {
        return this.model.getPageProps().action || 'view';
    },

    actionReactView(action) {
        if (!action) {
            return false;
        }

        if (action.indexOf('report-') !== -1) {
            const parts = action.split('-');

            action = parts[0];
        }

        const contentConfig = this.options.contentConfig[action];

        return contentConfig.contentReactViewClass;
    },

    destroyReactView() {
        if (this.content && this.content.$el) {
            ReactDOM.unmountComponentAtNode(this.content.$el[0]);
        }
    },

    showLoader() {
        if (!_.isString(this.ui.loader)) {
            this.ui.loader.show();
        }
    },

    hideLoader() {
        if (!_.isString(this.ui.loader)) {
            this.ui.loader.hide();
        }
    },

    runUpdate() {
        const currentTask = this.model.get('task');
        const currentStatus = currentTask && currentTask.get('status');
        const currsentConfig = this.options.contentConfig[this.getAction()];
        const canUseSimpleAutoUpdate = currsentConfig && currsentConfig.canBeUpdated;

        const self = this;

        if (!canUseSimpleAutoUpdate) {
            return;
        }

        self.model.getTask()
            .fetch({ reset: true })
            .done((model, status) => {
                if (model && status === 'success' && currentStatus !== model.status) {
                    Backbone.trigger('favicon:notify');
                    Backbone.trigger('notifications:push', self.model.getTask(), self.model.getPageProps());
                }

                app.reqres.request('AUTOUPDATE_SKIP', false);
                self.deActivateCommonEvents();
                self.showContent(false, true);
            }).fail(() => {
                self.deActivateCommonEvents();
                app.reqres.request('AUTOUPDATE_SKIP', false);
            });
    },

    activateAutoupdate(interval) {
        const self = this;

        if (self.autoUpdateIntervalID) {
            self.deActivateAutoupdate();
        }

        if (interval === BasicCollection.ONCE) {
            app.reqres.request('AUTOUPDATE_SKIP', true);
            self.runUpdate();
        } else if (interval) {
            self.autoUpdateIntervalID = setInterval(() => {
                if (!app.reqres.request('AUTOUPDATE_SKIP')) {
                    app.reqres.request('AUTOUPDATE_SKIP', true);
                    self.runUpdate();
                }
            }, (interval ? interval * 1000 : BasicCollection.AUTO_UPDATE_INTERVAL));
        }
    },

    deActivateAutoupdate() {
        app.reqres.request('AUTOUPDATE_SKIP', false);
        clearInterval(this.autoUpdateIntervalID);
    },

    activateEvents() {
        this.listenTo(Backbone, 'autoupdate:initialize', this.activateAutoupdate);
        this.listenTo(Backbone, 'autoupdate:on', this.activateAutoupdate);
        this.listenTo(Backbone, 'autoupdate:off', this.activateAutoupdate);
    },

    deActivateEvents() {
        this.stopListening(Backbone, 'autoupdate:initialize', this.activateAutoupdate);
        this.stopListening(Backbone, 'autoupdate:on', this.activateAutoupdate);
        this.stopListening(Backbone, 'autoupdate:off', this.activateAutoupdate);
    },

    deActivateCommonEvents() {
        this.deActivateEvents();

        if (this.model.getTask()) {
            this.model.getTask().offProps();
        }
    },

    onClose() {
        this.deActivateCommonEvents();
        this.deActivateAutoupdate();
        this.disableAutoupdateFaviconClient();
    },

    serializeData() {
        return {
            loaderMessage: this.options.loaderMessage
        };
    }
});

module.exports = SingleTaskContentLayout;
