const router = require('./../../../router');
const PresetManageView = require('./presetManageView/PresetManageView');

/**
 * @class PresetsView
 * @extends Backbone.Marionette.ItemView
 *
 * @property {CoreApplication} model
 * @property {UserPresetsCollection} collection
 */
const PresetsView = Marionette.ItemView.extend({

    tagName: 'td',
    className: 'c',

    template: require('./tpl/PresetsView.hbs'),

    ui: {
        button: '.button'
    },

    events: {
        'click .app_preset_trigger': 'onPresetSelected',
        'click .app_preset_manage': 'onPresetManageTrigger',
        'click .menu': 'onMenuItemClick'
    },

    collectionEvents: {
        'destroy remove add': 'render',
        change: 'debouncedRender'
    },

    visibleItemsQnt: 3,

    initialize() {
        this.debouncedRender = _.debounce(this.render, 100);

        setTimeout(() => {
            this.collection.updateCounters();
        }, 50);

        window.addEventListener('resize', _.throttle(this.render, 100));
    },

    onRender() {
        this.$('.button').bemmyButton();
    },

    onDestroy() {
        window.removeEventListener('resize', _.throttle(this.render, 100));
    },

    serializeData() {
        const serialized = {
            presets: this.serializePresets()
        };

        return serialized;
    },

    serializePresets() {
        let serialized = [];

        this.collection.forEach(preset => {
            let found = preset.get('found');

            found = (found > 99 ? '99+' : found);

            if (this.collection.presetForCurrentPage(preset.get('forPage'))) {
                serialized.push({
                    id: preset.get('id'),
                    title: preset.get('title'),
                    found,
                    order: preset.get('order')
                });
            }
        });

        serialized = _.sortBy(serialized, 'order');

        const visibleItemsQnt = this.getVisibleItemsQnt(serialized);

        return {
            main: serialized.slice(0, visibleItemsQnt),
            secondary: serialized.slice(visibleItemsQnt)
        };
    },

    getVisibleItemsQnt(items) {
        const OFFSETS = {
            sides: 445,
            foundDigit: 10 * 3,
            foundOffset: 6,
            letter: 11,
            more: 75
        };
        let width = document.body.clientWidth - OFFSETS.sides;
        let counter = 0;
        let variableLength;
        let ended = false;

        (items || []).forEach(item => {
            if (ended) {
                return;
            }

            variableLength = (OFFSETS.foundDigit + OFFSETS.foundOffset) +
                (OFFSETS.letter * (item.title || '').length);

            if (!item.title || (width > 0 && width - variableLength > 0)) {
                counter += 1;
                width -= variableLength;
            } else {
                ended = true;
            }
        });

        if (items.length > counter) {
            if (variableLength < OFFSETS.more) {
                counter -= 1;
            }
        }

        return counter;
    },

    onPresetSelected(evtObj) {
        evtObj.preventDefault();

        const presetId = parseInt(evtObj.target.dataset.id, 10);
        const preset = this.collection.where({ order: presetId })[0];

        if (preset) {
            router.navigate(
                '/' + preset.get('forPage') +
                '?' + router.buildQueryString(preset.serializeAsUrl()),
                { trigger: true }
            );
        }
    },

    onPresetManageTrigger(evtObj) {
        evtObj.preventDefault();

        (new PresetManageView({
            collection: this.collection
        })).show();
    },

    onMenuItemClick() {
        this.ui.button.bemmy().toggleHide();
    }
});

module.exports = PresetsView;
