const SecNavigation = require('./SecNavigation');
const BatchControlsView = require('../../controlSection/batchControls/BatchControlsView.js');

/**
 * @class SingleTaskSecNavigation
 * @extends SecNavigation
 *
 * @property {CoreApplication} model
 */
const SingleTaskSecNavigation = SecNavigation.extend({

    className: 'section__nav section__nav_bottom section__nav_nobd section__nav_red',

    template: require('./tpl/SingleTaskSecNavigation.hbs'),

    ui: {
        navItems: '.c:not(.external)',
        spinner: '.spinner'
    },

    options: _.extend({
        batchCtrlsToggler: 'section__nav_batch'
    }, SecNavigation.prototype.options),

    events: {
        'click @ui.navItems': 'onAction'
    },

    initialize() {
        SecNavigation.prototype.initialize.apply(this, arguments);

        const task = this.model.getTask();

        if (task instanceof Backbone.Model) {
            this.listenTo(task, 'change:children', this.bindChildTasksEvents);
            this.listenTo(task, 'change:audit', this.render);
        }
    },

    onAction(evtObj) {
        evtObj.preventDefault();

        const target = evtObj.currentTarget;

        if (target.classList.contains(this.options.disabledClass) ||
            target.classList.contains(this.options.activeClass)) {
            return false;
        }

        this.showSpinner(target);
        this.options.layout.trigger('content:replace', {
            action: target.dataset.action
        });
    },

    onDestroy() {
        SecNavigation.prototype.onDestroy.apply(this, arguments);

        this.stopListening(this.model.getTask(), 'change:children', this.bindChildTasksEvents);
        this.unbindChildTasksEvents();
    },

    bindChildTasksEvents() {
        const children = this.model.getTask().get('children');

        if (children instanceof Backbone.Collection) {
            this.listenTo(children, 'change:selected', this.toggleControls);
            this.controls.close();
            this.controls.show(new BatchControlsView({
                collection: children
            }));
        }
    },

    unbindChildTasksEvents() {
        const children = this.model.getTask().get('children');

        if (children instanceof Backbone.Collection) {
            this.stopListening(children, 'change:selected', this.toggleControls);
        }
    },

    toggleControls() {
        const children = this.model.getTask().get('children');

        this.$el.removeClass(this.options.batchCtrlsToggler);

        if (children.getSelected && children.getSelected().length > 0) {
            this.$el.addClass(this.options.batchCtrlsToggler);
        }
    },

    serializeData() {
        const task = this.model.getTask();

        return {
            historyItemsQnt: task.getHistoryItemsQnt(),
            logsQnt: task.getLogsQnt(),
            resourcesQnt: task.getResourcesQnt(),
            dependantQnt: task.getDependantQnt(),
            childrenQnt: task.getChildrenQnt(),
            dependOnQnt: task.getDependOnQnt(),
            contextQnt: task.getContextQnt(),
            hostsQnt: task.getHostsQnt(),
            id: task.get('id'),
            reports: task.getReportTabs(),
            isOriginalTask: this.model.get('pageProps').type === 'SINGLE_TASK'
        };
    }
});

module.exports = SingleTaskSecNavigation;
