const ServiceNotificationsMonitorItemView = require('./ServiceNotificationsMonitorItemView');
const ServiceNotificationsCollection = require('../../../collection/ServiceNotificationsCollection');
const ServiceNotificationPrefsModel = require('../../../model/ServiceNotificationPrefsModel');
const error = require('../../../error');

/**
 * @class ServiceNotificationsMonitorView
 * @extends Marionette.CollectionView
 */
const ServiceNotificationsMonitorView = Marionette.CollectionView.extend({

    id: 'service-notification',

    itemView: ServiceNotificationsMonitorItemView,

    collectionEvents: {
        change: 'onMarkedAsClosed'
    },

    initialize() {
        this._collection = new ServiceNotificationsCollection();
        this.collection = new Backbone.VirtualCollection(this._collection, { filter() {
            return false;
        } });
        this.closedItems = new ServiceNotificationPrefsModel();

        this.getServiceNotifications();
        this.startPolling();
    },

    startPolling() {
        const self = this;

        this.pollingInterval = setInterval(() => {
            self._collection.fetch({ reset: true });
        }, 30000);
    },

    getServiceNotifications() {
        const self = this;

        this._collection
            .fetch({ reset: true })
            .done(() => {
                self.closedItems
                    .fetch({ reset: true })
                    .always(() => {
                        if (self.closedItems.isEmpty()) {
                            self.setTrulyFilter();
                        } else {
                            self.setNotClosedFilter();
                        }
                    });
            })
            .fail(() => {
                error.message('Can not load service notifications', { reload: false });
            });
    },

    setTrulyFilter() {
        this.collection.updateFilter(() => {
            return true;
        });
    },

    setNotClosedFilter() {
        const self = this;

        this.collection.updateFilter(notification => {
            return !self.closedItems.isClosed(notification.get('id'));
        });
    },

    onClose() {
        clearInterval(this.pollingInterval);
    },

    onMarkedAsClosed(model) {
        if (model.get('markedAsClosed')) {
            this.closedItems.addClosed(model.get('id'));
        }
    }
});

module.exports = ServiceNotificationsMonitorView;
