/* eslint max-statements: [1, 11] */
const React = require('react');
const ReactDOM = require('react-dom');

const ResourceAttributesView = require('./attributesView/ResourceAttributesView');
// eslint-disable-next-line no-unused-vars
const SingleTaskSideNavView = require('../singleTask/SingleTaskSideNavView');
const ClipboardClientMixin = require('../components/clipboardClient/ClipboardClientMixin');
const SingleTaskCommonView = require('../singleTask/SingleTaskCommonView');
const ResourceLinksView = require('./ResourceLinksView');
const Timer = require('../../helpers/utils/Timer.js');
// eslint-disable-next-line no-unused-vars
const notification = require('../../notification');
// eslint-disable-next-line no-unused-vars
const error = require('../../error');
// eslint-disable-next-line no-unused-vars
const app = require('../../app');

const OwnerInfo = require('../../components/OwnerInfo');

/**
 * @class SingleResourceCommonView
 * @mixes ClipboardClientMixin
 * @extends SingleTaskCommonView
 *
 * @property {TaskResourceModel} model
 * @property {Marionette.Region} httpLinks
 * @property {Marionette.Region} rsyncLinks
 */
const SingleResourceCommonView = SingleTaskCommonView.mixin(ClipboardClientMixin).extend({

    ui: {
        ownerAsideInfo: '#owner-info'
    },

    template: require('./tpl/compact/SingleResourceCommonView.hbs'),

    modelEvents: {
        sync: 'showAttributesRegion',
        actionSuccess: 'fetchModel'
    },

    fetchModel() {
        this.model.fetch({ reset: true });
    },

    onRender() {
        this.initClipboardClient();

        this.showAttributesRegion();
        this.showLinkRegions();

        if (this.ui.ownerAsideInfo[0]) {
            ReactDOM.render(
                <OwnerInfo owner={this.model.get('owner')} namespace={'schedulers'}/>,
                this.ui.ownerAsideInfo[0]
            );
        }

        $('.input').bemmyInput();
    },

    initClipboardClient() {
        this.initializeClipboardClient(
            '.skynet-copier',
            'Resource Skynet copier ID has been copied to clipboard.'
        );

        this.initializeClipboardClient(
            '.http-proxy-copier',
            'Resource HTTP proxy link has been copied to clipboard.'
        );
    },

    showAttributesRegion() {
        this.attributes.close();
        this.attributes.show(new ResourceAttributesView({
            model: this.model,
            collection: new Backbone.Collection(this.model.serializeAttrs())
        }));
    },

    showLinkRegions() {
        const http = this.model.get('http');
        const rsync = this.model.get('rsync');

        if (http.links.length && this.httpLinks) {
            this.httpLinks.close();
            this.httpLinks.show(new ResourceLinksView({
                type: 'http',
                model: this.model,
                collection: new Backbone.Collection(_.map(http.links, link => {
                    return { link };
                })),
                linkable: true,
                actionBtn: true
            }));
        }

        if (rsync.links.length && this.rsyncLinks) {
            this.rsyncLinks.close();
            this.rsyncLinks.show(new ResourceLinksView({
                type: 'rsync',
                model: this.model,
                collection: new Backbone.Collection(_.map(rsync.links, link => {
                    return { link };
                }))
            }));
        }
    },

    serializeData() {
        const serialized = this.model.toJSON();

        serialized.created = this.serializeCreated();
        serialized.expires = serialized.time.expires ?
            Timer.serializeTimeMark(Timer.toLocal(serialized.time.expires)) :
            { comment: 'Never' };
        serialized.accessed = Timer.serializeTimeMark(serialized.time.accessed);
        serialized.attributes = this.model.serializeAttrs();
        serialized.canBeChanged = this.model.canBeChanged();

        return serialized;
    }
});

module.exports = SingleResourceCommonView;
