const React = require('react');
const ReactDOM = require('react-dom');

const TaskModel = require('../../model/TaskModel');
const TaskResources = require('../../collection/TaskResourcesCollection');
const BasicCollection = require('../../collection/BasicCollection');
const BasicTaskLayout = require('../tasks/tasksGrid/row/BasicTaskLayout');
const SingleTaskCopiedFrom = require('./SingleTaskCopiedFrom');
const SingleTaskProcessMonitor = require('./SingleTaskProcessMonitor');
const SingleTaskRefreshableInfo = require('./SingleTaskRefreshableInfo');
const SingleTaskDescriptionView = require('./SingleTaskDescriptionView');
const SingleTaskCustomHeaderView = require('./SingleTaskCustomHeaderView');
const SingleTaskCustomFooterView = require('./SingleTaskCustomFooterView');
const SingleTaskReleasedResources = require('./SingleTaskReleasedResources');
const SingleTaskCommonCustomFieldsView = require('./customProps/SingleTaskCommonCustomFieldsView');
const SingleTaskCommonCustomOutputFieldsView = require('./customProps/SingleTaskCommonCustomOutputFieldsView');
const HostsInfo = require('../../components/HostsInfo');
const AdvancedInfo = require('../../components/AdvancedInfo');
const SemaphoreInfo = require('../../components/SemaphoreInfo');
const ClientTagsInfo = require('../../components/ClientTagsInfo');
const NotificationInfo = require('../../components/NotificationInfo');
const OwnerInfo = require('../../components/OwnerInfo');

const currentActionTpl = require('./tpl/compact/currentAction.hbs');

/**
 * @class SingleTaskCommonLayout
 * @extends BasicTaskLayout
 *
 * @property {TaskModel} model
 * @property {Marionette.Region} hostsInfo
 * @property {Marionette.Region} description
 * @property {Marionette.Region} notification
 * @property {Marionette.Region} taskProcesses
 * @property {Marionette.Region} refreshableData
 * @property {Marionette.Region} releasedResources
 * @property {Marionette.Region} copiedFrom
 * @property {Marionette.Region} customFields
 * @property {Marionette.Region} customOutputFields
 * @property {Marionette.Region} customFooter
 */
const SingleTaskCommonView = BasicTaskLayout.extend({

    className: 'section__side',
    template: require('./tpl/compact/SingleTaskCommonView.hbs'),

    ui: {
        logsCnt: '.logs',
        logsTrigger: '.logs__ctrl a',
        currentActionCnt: '.curr_action_cnt',
        advancedAsideInfo: '#advancedAsideInfo',
        semaphoresAsideInfo: '#semaphoresAsideInfo',
        ownerAsideInfo: '#owner-info',
        notificationAsideInfo: '#notificationAsideInfo',
        clientTagsInfo: '#clientTagsInfo',
        hostsInfo: '#hostsInfo'
    },

    events: {
        'click @ui.logsTrigger': 'toggleLogs'
    },

    modelEvents: {
        sync: 'onModelSync'
    },

    initialize() {
        setTimeout(() => {
            Backbone.trigger(
                'autoupdate:initialize',
                Math.floor(BasicCollection.FORCE_AUTO_UPDATE_INTERVAL / 1000)
            );
        }, 0);

        BasicTaskLayout.prototype.initialize.call(this);
    },

    onRender() {
        /* eslint max-statements: [1, 37] */
        const statesForProcessMonitor = ['executing', 'preparing', 'assigned', 'finishing', 'suspended', 'suspending'];

        if (statesForProcessMonitor.includes(this.model.get('status').toLowerCase())) {
            this.taskProcesses.close();
            this.taskProcesses.show(new SingleTaskProcessMonitor({
                model: this.model
            }));
        }

        this.renderRefreshableData();

        this.description.close();
        this.description.show(new SingleTaskDescriptionView({
            model: this.model
        }));

        if (this.shouldShowReleaseInfo()) {
            const taskReleaseInfo = this.model.get('release');
            let releaseResources = [];

            if (taskReleaseInfo && taskReleaseInfo.resources) {
                releaseResources = taskReleaseInfo.resources;
            }

            if (this.releasedResources) {
                this.releasedResources.close();
                this.releasedResources.show(new SingleTaskReleasedResources({
                    model: this.model,
                    collection: new TaskResources(_.map(releaseResources, resource => {
                        return { id: resource.resource_id };
                    }), {})
                }));
            }
        }

        this.copiedFrom.close();
        this.copiedFrom.show(new SingleTaskCopiedFrom({
            model: this.model
        }));

        this.customFields.close();
        this.customFields.show(new SingleTaskCommonCustomFieldsView({
            model: this.model,
            taskProps: this.options.taskProps
        }));

        this.customOutputFields.close();
        this.customOutputFields.show(new SingleTaskCommonCustomOutputFieldsView({
            model: this.model,
            taskProps: this.options.taskProps
        }));

        this.customHeader.close();
        this.customHeader.show(new SingleTaskCustomHeaderView({
            model: this.model
        }));

        this.customFooter.close();
        this.customFooter.show(new SingleTaskCustomFooterView({
            model: this.model
        }));

        if (this.ui.ownerAsideInfo[0]) {
            ReactDOM.render(
                <OwnerInfo owner={this.model.get('owner')} namespace={'tasks'}/>,
                this.ui.ownerAsideInfo[0]
            );
        }

        if (this.ui.hostsInfo[0]) {
            ReactDOM.render(
                <HostsInfo model={this.model}/>,
                this.ui.hostsInfo[0]
            );
        }

        if (this.ui.advancedAsideInfo[0]) {
            ReactDOM.render(
                <AdvancedInfo model={this.model}/>,
                this.ui.advancedAsideInfo[0]
            );
        }

        if (this.ui.clientTagsInfo[0]) {
            ReactDOM.render(
                <ClientTagsInfo model={this.model}/>,
                this.ui.clientTagsInfo[0]
            );
        }

        if (this.ui.semaphoresAsideInfo[0]) {
            ReactDOM.render(
                <SemaphoreInfo model={this.model}/>,
                this.ui.semaphoresAsideInfo[0]
            );
        }

        if (this.ui.notificationAsideInfo[0]) {
            ReactDOM.render(
                <NotificationInfo model={new Backbone.Model({ notifications: this.model.get('notifications') })}/>,
                this.ui.notificationAsideInfo[0]
            );
        }
    },

    onModelSync() {
        /* eslint complexity: [1, 12] */

        const prevStatus = this.model.previousAttributes().status;
        const status = this.model.get('status');

        this.updateCurrentAction();

        if (prevStatus !== status &&
            (status === TaskModel.STATUS.EXECUTING ||
            status === TaskModel.STATUS.FINISHED ||
            status === TaskModel.STATUS.FAILURE ||
            status === TaskModel.STATUS.SUCCESS ||
            status === TaskModel.STATUS.EXCEPTION ||
            status === TaskModel.STATUS.SUSPENDED ||
            status === TaskModel.STATUS.NO_RES ||
            status === TaskModel.STATUS.TIMEOUT ||
            status === TaskModel.STATUS.EXPIRED ||
            status === TaskModel.STATUS.WAIT_CHILD)) {
            this.render();
        } else {
            this.renderRefreshableData();
        }
    },

    shouldShowReleaseInfo() {
        const taskReleaseInfo = this.model.get('release');

        return (
            [TaskModel.STATUS.RELEASED, TaskModel.STATUS.SUCCESS].includes(this.model.get('status')) &&
            taskReleaseInfo &&
            taskReleaseInfo.resources
        );
    },

    toggleLogs(evtObj) {
        evtObj.preventDefault();
        this.ui.logsCnt.toggleClass('opened');
    },

    renderRefreshableData() {
        this.refreshableData.close();
        this.refreshableData.show(new SingleTaskRefreshableInfo({
            model: this.model
        }));
    },

    serializeData() {
        let serialized = BasicTaskLayout.prototype.serializeData.apply(this, arguments);

        if (serialized.logs) {
            serialized.moreLogs = serialized.logs.slice(6);
            serialized.logs = serialized.logs.slice(0, 6);
        }

        if (serialized.scheduler) {
            serialized.scheduler.icon = 'images/icons/restore.svg';

            if (serialized.scheduler.id < 0) {
                serialized.scheduler.icon = 'images/icons/play-icon.svg';
                serialized.scheduler.id *= -1;
            }
        }

        serialized.hasReleaseInfo = this.shouldShowReleaseInfo();
        serialized = this.checkDiskExceeded(serialized);

        return serialized;
    },

    checkDiskExceeded(serialized) {
        const results = serialized.results;
        const requirements = serialized.requirements;

        if (!results || !results.disk_usage) {
            return serialized;
        }

        const space = {
            used: results.disk_usage && results.disk_usage.max,
            free: requirements.disk_space
        };

        const isSpaceClosedToExceed = space.used >= space.free * 0.95 && space.used <= space.free;
        const isSpaceExceeded = space.used > space.free;

        if (isSpaceClosedToExceed) {
            results.disk_usage.error = {
                type: 'warn',
                text: 'Task may exceed disk requirements'
            };
        }

        if (isSpaceExceeded) {
            results.disk_usage.error = {
                type: 'crit',
                text: 'Task exceeded disk requirements'
            };
        }

        return serialized;
    },

    updateCurrentAction() {
        const execution = this.model.get('execution');

        if (this.ui.currentActionCnt) {
            if (execution && execution.action && execution.action.message) {
                this.ui.currentActionCnt.show();
                this.ui.currentActionCnt.html(currentActionTpl({
                    message: execution.action.message,
                    duration: execution.action.duration
                }));
            } else {
                this.ui.currentActionCnt.hide();
            }
        }
    }
});

module.exports = SingleTaskCommonView;
