const sanitize = require('sanitize-html');
const error = require('../../error');

const opts = {
    allowedTags: sanitize.defaults.allowedTags.concat(['style', 'input', 'span']),
    allowedAttributes: Object.assign(
        {},
        sanitize.defaults.allowedAttributes,
        {
            input: ['name', 'value', 'checked', 'style', 'type'],
            td: ['colspan', 'rowspan'],
            th: ['colspan', 'rowspan'],
            '*': ['style', 'color', 'bgcolor', 'title', 'valign', 'class']
        }
    )
};

/**
 * @class SingleTaskDescriptionView
 * @extends Marionette.ItemView
 *
 * @property {TaskModel} model
 */
const SingleTaskDescriptionView = Marionette.ItemView.extend({

    className: 'monitor__g monitor_description',
    template: require('./tpl/SingleTaskDescriptionForm.hbs'),

    ui: {
        form: '.task_descr textarea',
        applyDesc: '.apply_description',
        cancelDesc: '.cancel_description',
        modeToggler: '.mode_toggler'
    },

    events: {
        'input @ui.form': 'onDescriptionChange',
        'click @ui.modeToggler': 'changeMode',
        'click @ui.cancelDesc': 'changeMode',
        'click @ui.applyDesc': 'onDescriptionApply'
    },

    initialize() {
        this.options.editMode = false;
    },

    onRender() {
        this.$('.textarea').bemmyTextarea();
    },

    serializeData() {
        return {
            editMode: this.options.editMode,
            description: this.model.get('description')
        };
    },

    changeMode() {
        this.options.editMode = !this.options.editMode;
        this.render();
    },

    onDescriptionApply() {
        if (this.ui.form.val() !== '') {
            this.model.set('description', sanitize(this.ui.form.val(), opts));

            if (this.model.scheduler) {
                $.ajax({
                    url: ('/api/v1.0/scheduler/' + this.model.scheduler.get('id')),
                    type: 'PUT',
                    data: JSON.stringify({ task: { description: this.model.get('description') }}),
                    dataType: 'json',
                    contentType: 'application/json'
                }).fail(error.fromXHR);
            } else {
                $.ajax({
                    url: ('/api/v1.0/task/' + this.model.get('id')),
                    type: 'PUT',
                    data: JSON.stringify({ description: this.model.get('description') }),
                    dataType: 'json',
                    contentType: 'application/json'
                }).fail(error.fromXHR);
            }

            this.changeMode();
        }
    },

    onDescriptionChange() {
        if (this.ui.form.val() === '') {
            this.ui.applyDesc.addClass('button_disable');
        } else {
            this.ui.applyDesc.removeClass('button_disable');
        }
    }
});

module.exports = SingleTaskDescriptionView;
