/**
 * @class SingleTaskHistoryItemsView
 * @extend Backbone.Marionette.ItemView
 */
const SingleTaskHistoryItemsView = Marionette.ItemView.extend({

    tagName: 'table',
    className: 't t_max',
    id: 'history',

    timeFormatting: -1,

    template: require('./tpl/SingleTaskHistoryItemsView.hbs'),

    ui: {
        childrenCnt: '.t__child',
        childrenCntToggler: '.t__child__ctrl'
    },

    options: {
        childrenCntToggleClass: 't__child_open',
        childrenCntButtonToggleClass: 'button_state_check'
    },

    events: {
        'click @ui.childrenCntToggler': 'toggleChildrenCnt'
    },

    collectionEvents: {
        'add remove update': 'render',
        'format:timing': 'onChangeFormat'
    },

    behaviors: {
        ClickOut: {}
    },

    initialize(options) {
        this.timeFormatting = options.timeFormatting ||
            SingleTaskHistoryItemsView.TIME_FORMATTING.RELATIVE_TO_CREATE;
    },

    onClickOut() {
        this.toggleChildrenCnt(false);
    },

    toggleChildrenCnt(toggle) {
        if (this.ui.childrenCnt) {
            const button = this.ui.childrenCnt.find('.button_child-show');

            this.ui.childrenCnt.toggleClass(this.options.childrenCntToggleClass, toggle);

            button.toggleClass(this.options.childrenCntButtonToggleClass);
        }
    },

    onChangeFormat(format) {
        this.timeFormatting = format;
        this.render();
    },

    serializeData() {
        const rows = this.collection.toJSON();

        rows.forEach(function (row, index) {
            /* eslint max-statements: [1, 11] */
            /* eslint camelcase: 0 */

            let indexOfRelativeTask;

            if (this.timeFormatting === SingleTaskHistoryItemsView.TIME_FORMATTING.RELATIVE_TO_PREVIOUS) {
                if (index === 0) {
                    indexOfRelativeTask = 0;
                } else {
                    indexOfRelativeTask = index - 1;
                }
            } else {
                indexOfRelativeTask = 0;
            }

            if (row.status) {
                row.status = row.status.toLowerCase();
            }

            if (row.iface) {
                row.iface = row.iface.toUpperCase();
            }

            row.s_time = this.serializeTime(row.time, rows[indexOfRelativeTask].time);
        }, this);

        return {
            rows,
            isAdmin: this.options.isAdmin
        };
    },

    /**
     * Serialize time
     * @param {Number}  timeTask
     * @param {Number}  timeTaskRelative
     * @returns {{main: String, secondary: String}}
     */
    serializeTime(timeTask, timeTaskRelative) {
        const date = moment(timeTask);
        const absolute = {
            main: date.format('HH:mm:ss'),
            secondary: date.format('DD.MM.YYYY')
        };
        let serialized;

        switch (this.timeFormatting) {
            case SingleTaskHistoryItemsView.TIME_FORMATTING.RELATIVE_TO_NOW: {
                serialized = {
                    absolute,
                    main: date.fromNow(),
                    secondary: ''
                };

                break;
            }
            default: {
                if (date.isSame(timeTaskRelative)) {
                    serialized = {
                        absolute,
                        main: date.fromNow(),
                        secondary: ''
                    };
                } else {
                    serialized = {
                        absolute,
                        main: moment.duration(date.diff(timeTaskRelative)).humanize(),
                        secondary: '+'
                    };
                }
            }
        }

        return serialized;
    }
});

SingleTaskHistoryItemsView.TIME_FORMATTING = {
    RELATIVE_TO_CREATE: 2,
    RELATIVE_TO_PREVIOUS: 3,
    RELATIVE_TO_NOW: 4
};

module.exports = SingleTaskHistoryItemsView;
