const BasicCollection = require('../../collection/BasicCollection');
const sanitize = require('sanitize-html');
const promisedXhr = require('../../helpers/utils/promisedXhr');

/**
 * @class SingleTaskProcessMonitor
 * @extends Marionette.ItemView
 *
 * @property {TaskModel} model
 */
const SingleTaskProcessMonitor = Marionette.ItemView.extend({

    className: 'monitor monitor_dark',

    template: require('./tpl/SingleTaskProcessMonitor.hbs'),

    ui: {
        processesQnt: '.prcQnt',
        monitorCnt: '.monitor_container',
        interval: 'select',
        shell: '.open_shell'
    },

    events: {
        'click @ui.shell': 'onShellOpen'
    },

    initialize() {
        this.updateInterval = BasicCollection.FORCE_AUTO_UPDATE_INTERVAL;
        this.processes = '';
        this.updateId = null;

        this.updateProcesses = this.updateProcesses.bind(this);
    },

    onRender() {
        this.$('.select').bemmySelect();

        if (this.updateInterval) {
            this.startUpdatingProcesses();
        }
    },

    onShellOpen() {
        const executionData = this.model.get('execution');
        const shellUrl = executionData && executionData.shell_url;
        const clientId = executionData && executionData.client.id;

        if (shellUrl) {
            const left = (screen.availWidth / 2) - 400;
            const top = (screen.availHeight / 2) - 200;
            const shell = window.open(
                shellUrl,
                clientId,
                'resizable=yes,scrollbars=yes,location=no,height=400,width=800,left=' + left + ',top=' + top
            );

            shell.focus();
        }
    },

    startUpdatingProcesses() {
        this.updateId = setInterval(this.updateProcesses, this.updateInterval);
    },

    updateProcesses() {
        const self = this;

        /* eslint dot-notation: 0 */

        promisedXhr(this.model.get('execution')['ps_url'])
            .then(data => {
                const lines = data.match(/\n/gm);
                const processData = sanitize(data, {
                    allowedTags: [],
                    textFilter(text) {
                        return text
                            .replace(/&amp;/, '&')
                            .replace(/&lt;/, '<')
                            .replace(/&gt;/, '>');
                    }
                });

                if (processData.trim().length) {
                    self.ui.monitorCnt.text(processData);
                    self.ui.processesQnt.text(lines ? lines.length - 1 : 0);
                }
            })
            .catch(err => {
                /* eslint no-console: 0 */
                console.error('xhr error!', err.statusText);
            });
    },

    serializeData() {
        return {
            runnedProcessed: 0,
            updateInterval: this.updateInterval,
            processes: this.processes,
            shellShowed: this.model.get('status').toLowerCase() === 'suspended'
        };
    },

    onClose() {
        if (this.updateId) {
            clearInterval(this.updateId);
        }
    }
});

module.exports = SingleTaskProcessMonitor;
