/* eslint guard-for-in: 0 */

const ChartPlaceholder = require('./ChartPlaceholder');
const rpsChartTooltip = require('./tpl/RPSChartTooltip.hbs');

/**
 * @class RPSChartView
 * @extends ChartPlaceholder
 */
const RPSChartView = ChartPlaceholder.extend({

    initialize(options) {
        this.options = _.extend({}, ChartPlaceholder.prototype.options, {
            chartOptions: {
                credits: { text: 'Yandex.com', href: 'http://company.yandex.com/' },
                title: { text: options.specifiedChartName, margin: 25 },
                xAxis: {
                    categories: [],
                    labels: { rotation: -45 }
                },
                yAxis: [
                    { // Primary yAxis
                        labels: { format: '{value} RPS' },
                        title: { text: 'RPS' },
                        minorTickInterval: 'auto',
                        stackLabels: {
                            x: 3,
                            enabled: true,
                            rotation: -90,
                            textAlign: 'left',
                            style: { fontWeight: 'bold' },
                            formatter() {
                                return Highcharts.numberFormat(this.total, 0, '.', ' ');
                            }
                        },
                        plotLines: [{
                            value: 0,
                            color: 'red',
                            width: 1,
                            zIndex: 5,
                            label: {
                                text: 'Average RPS: ',
                                align: 'center',
                                style: { color: 'gray', fontWeight: 'bold', 'text-shadow': '1px 1px 0px white' }
                            }
                        }]
                    },
                    { // Secondary yAxis
                        title: { text: 'Amount, requests' },
                        stackLabels: {
                            x: 3,
                            enabled: true,
                            rotation: -90,
                            textAlign: 'left',
                            style: { fontWeight: 'bold' },
                            formatter() {
                                return Highcharts.numberFormat(this.total, 0, '.', ' ');
                            }
                        },
                        opposite: true,
                        plotLines: [{
                            value: 0,
                            color: 'red',
                            width: 1,
                            zIndex: 5,
                            label: {
                                text: 'Average Amount: ',
                                align: 'center',
                                style: { color: 'gray', fontWeight: 'bold', 'text-shadow': '1px 1px 0px white' }
                            }
                        }]
                    }
                ],
                legend: {
                    x: 0,
                    y: 10,
                    align: 'center',
                    verticalAlign: 'top',
                    layout: 'horizontal',
                    floating: true,
                    borderWidth: 0
                },
                tooltip: {
                    formatter() {
                        return rpsChartTooltip(this);
                    }
                },
                plotOptions: {
                    column: {
                        shadow: true,
                        animation: true,
                        stacking: 'normal',
                        dataLabels: { enabled: false }
                    }
                },
                series: []
            }
        }, options);

        this.options.chartOptions = _.extend(
            {},
            ChartPlaceholder.prototype.options.chartOptions,
            this.options.chartOptions
        );
    },

    onRender() {
        this.ui.chartContainer.highcharts(this.preProcessData(this.options.data));
    },

    preProcessData(data) {
        /* eslint no-redeclare: 0 */
        /* eslint max-statements: [1, 33] */
        /* eslint complexity: [1, 7] */

        let i = 0;
        let max = { len: 0, hour: null };
        const options = _.clone(this.options.chartOptions);

        for (const hour in data) {
            const len = Object.keys(data[hour]).length;

            if (max.len < len) {
                max = { len, hour };
            }
        }

        const hd = data[max.hour];

        const series = new Array(Object.keys(hd).length * 2);

        for (const srv in hd) {
            series[i * 2] = {
                name: srv.replace('sandbox-', ''),
                type: 'column',
                stack: 'rps',
                tooltip: ' RPS',
                color: options.colors[i],
                data: []
            };

            series[(i * 2) + 1] = {
                name: srv.replace('sandbox-', ''),
                type: 'column',
                stack: 'requests',
                tooltip: ' requests',
                color: options.colors[i],
                data: [],
                yAxis: 1,
                showInLegend: false
            };
            ++i;
        }

        const categories = [];
        const total = { rps: 0, amount: 0 };

        for (const hour in data) {
            const hd = data[hour];

            categories.push(hour);

            i = 0;

            for (const srv in hd) {
                const shd = hd[srv] || { rps: 0, amount: 0 };

                total.rps += shd.rps;
                total.amount += shd.amount;
                series[i * 2].data.push(shd.rps);
                series[(i * 2) + 1].data.push(shd.amount);
                ++i;
            }
        }

        options.xAxis.categories = categories;

        options.yAxis[0].plotLines[0].value = (total.rps / categories.length).toFixed(2);
        options.yAxis[0].plotLines[0].label.text += options.yAxis[0].plotLines[0].value;
        options.yAxis[1].plotLines[0].value = (total.amount / categories.length).toFixed(2);
        options.yAxis[1].plotLines[0].label.text += options.yAxis[1].plotLines[0].value;

        options.series = series;

        return options;
    }

});

module.exports = RPSChartView;
