/* eslint guard-for-in: 0 */

const RPSChartView = require('./RPSChartView');
const rpsChartTooltip = require('./tpl/RPSChartTooltip.hbs');
const ChartPlaceholder = require('./ChartPlaceholder');

/**
 * @class TopClientsChartView
 * @extends RPSChartView
 */
const TopClientsChartView = RPSChartView.extend({

    initialize(options) {
        this.options = _.extend(
            {},
            ChartPlaceholder.prototype.options,
            { chartOptions: TopClientsChartView.RPS_CHART_BASIC_OPTIONS },
            options
        );

        this.options.chartOptions = _.extend(
            {},
            ChartPlaceholder.prototype.options.chartOptions,
            this.options.chartOptions
        );
    },

    preProcessData(data) {
        const categories = Object.keys(data);

        categories.sort((a, b) => {
            return data[b].amount - data[a].amount;
        });

        if (data[categories[0]].amount - data[categories[3]].amount > 25000) {
            this.options.chartOptions.yAxis[0].type = 'logarithmic';
            this.options.chartOptions.yAxis[0].title.text += ' (logarithmic)';
        }

        return TopClientsChartView.processRpsData(
            _.extend(
                {},
                _.clone(this.options.chartOptions),
                { title: { text: this.options.specifiedChartName }}
            ),
            categories, data
        );
    }

}, {

    /**
     * @description RPS charts data processing logic. Copy-pasted AS IS from the current pages.
     *
     * @param {Object}   options
     * @param {Array}    categories
     * @param {Object[]} data
     *
     * @returns {Objects}
     */
    processRpsData(options, categories, data) {
        /* eslint no-redeclare: 0 */
        /* eslint max-statements: [1, 26] */
        /* eslint complexity: [1, 14] */

        options = _.clone(options);
        const methods = {};

        options.title = { text: options.title.text };

        for (const i in categories) {
            const st = data[categories[i]];
            const stmethods = 'methods' in st ? st.methods : st;

            for (const meth in stmethods) {
                if (meth.includes('.__repr__')) {
                    continue;
                }

                if (!(meth in methods)) {
                    methods[meth] = { calls: new Array(categories.length), rps: new Array(categories.length) };
                }
            }
        }

        for (const i in categories) {
            let st = data[categories[i]];

            st = 'methods' in st ? st.methods : st;

            for (const meth in methods) {
                methods[meth].rps[i] = meth in st ? st[meth].rps : 0;
                methods[meth].calls[i] = meth in st ? st[meth].amount : 0;
            }
        }

        for (const i in categories) {
            categories[i] = categories[i].replace('.search.yandex.net', '');
        }

        options.xAxis.categories = categories;

        const keys = Object.keys(methods);

        for (let i = 0; i < keys.length; ++i) {
            options.series.push({
                type: 'column',
                name: keys[i],
                data: methods[keys[i]].calls,
                color: options.colors[i]
            });
        }

        for (let i = 0; i < keys.length; ++i) {
            options.series.push({
                type: 'spline',
                name: keys[i],
                data: methods[keys[i]].rps,
                yAxis: 1,
                showInLegend: false,
                color: options.colors[i]
            });
        }

        return options;
    },

    RPS_CHART_BASIC_OPTIONS: {
        title: { text: '-- default title --' },
        credits: { text: 'Yandex.com', href: 'http://company.yandex.com/' },
        xAxis: {
            categories: [],
            labels: { rotation: -45 }
        },
        yAxis: [
            { // Primary yAxis
                title: { text: 'Amount, requests' },
                stackLabels: {
                    x: 3,
                    enabled: true,
                    rotation: -90,
                    textAlign: 'left',
                    style: { fontWeight: 'bold' },
                    formatter() {
                        return parseInt(this.total, 10);
                    }
                },
                minorTickInterval: 'auto',
                min: 1
            },
            { // Secondary yAxis
                labels: { format: '{value} RPS' },
                title: { text: 'RPS' },
                opposite: true,
                min: 0.01
            }
        ],
        tooltip: {
            formatter() {
                return rpsChartTooltip(this);
            }
        },
        plotOptions: {
            series: { enableMouseTracking: true },
            column: {
                shadow: true,
                animation: true,
                stacking: 'normal',
                dataLabels: { enabled: false }
            },
            spline: {
                lineWidth: 1.5,
                marker: { enabled: false },
                stacking: 'normal',
                animation: true
            }
        },
        series: []
    }
});

module.exports = TopClientsChartView;
