const BasicCollection = require('../../../collection/BasicCollection');

const BasicGridHeaderView = Marionette.ItemView.extend({

    tagName: 'thead',

    options: {
        ascSortedClass: 't__sort_up',
        descSortedClass: 't__sort_down',
        selectable: true,
        sortable: true,
        sortType: BasicCollection.SORT_TYPE.SERVER,
        sortKeys: []
    },

    ui: {
        sortblCl: '.t__sort',
        batchSelector: '.app_tasks_batch_select'
    },

    events: {
        'click @ui.sortblCl': 'onSortTasks',
        'bemmy:set @ui.batchSelector': 'onBatchSelect'
    },

    collectionEvents: {
        sync: 'render',
        sort: 'render',
        'change:selected': 'render'
    },

    initialize(options) {
        this.options = _.extend({}, this.options, options);
    },

    onRender() {
        this.$('.check').bemmyCheck();
    },

    onSortTasks(evtObj) {
        evtObj.preventDefault();

        const sortOrder = this.getSortOrder(evtObj.currentTarget.className);
        const sortKey = this.getSortKey(evtObj.currentTarget.dataset.sortType);

        this.collection.setSortField(sortOrder + sortKey, this.options.sortType);
        this.collection.setSelected(false);
    },

    onBatchSelect(evtObj, isSelected) {
        evtObj.preventDefault();

        this.collection.setSelected(isSelected);
    },

    /**
     * @memberOf TasksGridView
     * @description Default sort order is ASC
     * @param className
     * @returns {number}
     */
    getSortOrder(className) {
        let dir = BasicGridHeaderView.SORT_DIRS.ASC;

        if (className.includes(this.options.ascSortedClass)) {
            dir = BasicGridHeaderView.SORT_DIRS.DESC;
        }

        return dir;
    },

    /**
     * @memberOf TasksGridView
     * @param sortType
     * @returns {String} Empty string not possible key, should match @see Tasks.SORT_KEYS
     */
    getSortKey(sortType) {
        if (!sortType || !this.options.sortKeys.includes(sortType)) {
            sortType = '';
        }

        return sortType;
    },

    serializeData() {
        const serialized = {
            order: this.collection.getPagingOptions().order,
            sortable: this.options.sortable,
            expandable: this.options.expandable,
            unfilterable: this.options.unfilterable,
            selectable: this.options.selectable
        };

        if (this.options.selectable) {
            const selectedItemsLength = this.collection.getSelected().length;

            serialized.allChecked = (this.collection.length === selectedItemsLength && selectedItemsLength !== 0);
            serialized.partlyChecked = (this.collection.length !== selectedItemsLength && selectedItemsLength > 0);
        }

        return serialized;
    }
});

BasicGridHeaderView.SORT_DIRS = {
    ASC: '+',
    DESC: '-'
};

module.exports = BasicGridHeaderView;
