const Timer = require('../../../../helpers/utils/Timer.js');

/**
 * @class BasicTaskLayout
 * @extends Backbone.Marionette.Layout
 *
 * @description Utilize task serialization methods, that are used in
 *                  - {@link TaskRowView}
 *                  - {@link SingleTaskCommonLayout}
 *
 * @property {TaskModel} model
 */
const BasicTaskLayout = Marionette.Layout.extend({

    serializeData() {
        /* eslint max-statements: [1, 14] */

        const serialized = _.clone(this.model.attributes);

        serialized.status = serialized.status.toLowerCase();
        serialized.context = this.serializeContext();
        serialized.children = this.serializeChildren();
        serialized.execution = this.serializeExecutionInfo();

        if (this.model.get('time')) {
            serialized.created = this.serializeCreated();
            serialized.updated = this.serializeUpdated();
        }

        serialized.canBeEdited = this.model.canBeEdited();
        serialized.canBeStopped = this.model.canBeStopped();
        serialized.canBeDeleted = this.model.canBeDeleted();
        serialized.canBeExecuted = this.model.canBeExecuted();
        serialized.canBeClonned = this.model.canBeClonned();

        return serialized;
    },

    serializeChildren() {
        const children = this.model.get('children');
        const serialized = [];

        if ((children instanceof Array) || (children instanceof Backbone.Collection)) {
            children.forEach(child => {
                serialized.push(child instanceof Backbone.Model ? {
                    id: child.get('id'),
                    status: child.get('status').toLowerCase()
                } : child);
            });
        }

        return serialized;
    },

    /**
     * Format execution time
     * @param {Object}   Moment duration object
     * @returns {String}
     */
    formatExecutionTime(duration) {
        function leadZero(num) {
            return num < 10 ? ('0' + num) : num;
        }

        const days = duration.days() && (duration.days() + 'd:');
        const hours = duration.hours() && (leadZero(duration.hours()) + 'h:');
        const minutes = leadZero(duration.minutes()) + 'm';
        const seconds = leadZero(duration.seconds()) + 's';

        if (days) {
            return days + hours + minutes;
        }
        if (hours) {
            return hours + minutes;
        }
        return minutes + ':' + seconds;
    },

    serializeExecutionInfo() {
        const exec = this.model.get('execution');
        const current = moment.duration(exec.current, 's');
        let serialized = null;

        if (exec !== null) {
            const progress = exec.current * 100 / exec.estimated;

            serialized = {
                time: this.formatExecutionTime(current),
                action: exec.action,
                progress: (progress > 100 ? 100 : progress),
                client: (exec.client ? exec.client : {}),
                msg: exec.estimated && ((exec.estimated + 1) < exec.current) ?
                    'About ' + moment.duration(exec.current - exec.estimated, 'seconds').humanize() + ' overdue..' :
                    ''
            };
        }

        return serialized;
    },

    serializeCreated() {
        return Timer.serializeTimeMark(this.model.get('time').created);
    },

    serializeUpdated() {
        return Timer.serializeTimeMark(this.model.get('time').updated);
    },

    serializeContext() {
        const customFields = this.model.get('customFields');
        const customFieldsNames = customFields ?
            customFields.map(cField => {
                return cField.name;
            }) :
            [];
        const context = this.model.get('context');
        let result = [];

        if (context) {
            result = _.sortBy(Object.keys(context).map(key => {
                return {
                    key,
                    type: key === '__author' || key === 'notify_if_failed' || key === 'full_descr' ?
                        't__pre' :
                        '',
                    value: (typeof context[key] === 'object' ? JSON.stringify(context[key]) : context[key]),
                    isInputParam: (customFieldsNames.includes(key))
                };
            }), contextItem => {
                return contextItem.key.toLowerCase();
            });
        }

        return result;
    }
});

module.exports = BasicTaskLayout;
