const Timer = require('../../../../helpers/utils/Timer');
const TaskModel = require('../../../../model/TaskModel');
const TaskHistoryItemsCollection = require('../../../../collection/TaskHistoryItemsCollection');

/**
 * @class TaskHistoryView
 * @extends Marionette.ItemView
 *
 * @property {TaskModel}                  model
 * @property {TaskHistoryItemsCollection} collection
 */
const TaskHistoryView = Marionette.ItemView.extend({

    className: 't__time__b spinner_hidden',

    template: require('./tpl/TaskHistoryView.hbs'),

    events: {
        'get:history': 'onGetHistory'
    },

    collectionEvents: {
        sync: 'onFetchHistory'
    },

    options: {
        statuses: TaskModel.getPossibleStatuses()
    },

    initialize() {
        this.listenToOnce(this, 'get:history', this.onGetHistory);
        this.collection = new TaskHistoryItemsCollection([], { task: this.model, comparator: 'time' });
    },

    onGetHistory() {
        this.collection.fetch({ reset: true });
    },

    onFetchHistory() {
        this.render();
    },

    serializeData() {
        const self = this;
        const serialized = {
            milestones: [],
            loader: (this.collection.length < 2),
            id: this.model.get('id')
        };

        this.collection.forEach(milestone => {
            serialized.milestones.push({
                offset: (Timer.toLocal(milestone.get('time')).fromNow()),
                status: self.serializeStatus(milestone.get('status')),
                message: milestone.get('message')
            });
        });

        return serialized;
    },

    serializeStatus(status) {
        let serialized = status;
        const match = this.options.statuses.indexOf((status || '').toUpperCase());

        if (match !== -1) {
            serialized = (require('./tpl/TaskHistoryItemStatusBadge.hbs'))({
                status: status.toLowerCase()
            });
        }

        return serialized;
    }
});

module.exports = TaskHistoryView;
