const error = require('../../../../error');
const BasicTaskLayout = require('./BasicTaskLayout');
// eslint-disable-next-line no-unused-vars
const Timer = require('../../../../helpers/utils/Timer');
const TaskModel = require('../../../../model/TaskModel');
const TaskHistoryView = require('./TaskHistoryView');
const TaskSelectorView = require('./TaskSelectorView');
const TaskRowControlsView = require('./TaskRowControlsView');
const TaskImportantTogglerView = require('./TaskImportantTogglerView');

/**
 * @class TaskRowView
 * @extends Marionette.Layout
 * @property {TaskModel} model
 */
const TaskRowView = BasicTaskLayout.extend({

    template: require('./tpl/TaskRowView.hbs'),
    tagName: 'tr',

    options: {
        childrenCntToggleClass: 't__child_open',
        childrenCntButtonToggleClass: 'button_state_check',
        execStartClass: 'run',
        selectable: true
    },

    ui: {
        childrenCnt: '.t__child',
        taskHistoryToggler: '.app_task_history',
        childrenCntToggler: '.t__child__ctrl'
    },

    events: {
        'click @ui.childrenCntToggler': 'toggleChildrenCnt'
    },

    modelEvents: {
        sync: 'onModelSynced',
        error: 'onModelSyncError',
        'change:status': 'render',
        'change:_duplicating': 'render'
    },

    behaviors: {
        ClickOut: {}
    },

    /** REGIONS DEFINED DIRECTLY IN THE TEMPLATE */

    onRender() {
        const self = this;

        this.$('.check').bemmyCheck();
        this.$('.button').bemmyButton();
        this.$('.popup-overlay').bemmyPopupOverlay();

        this.showRegions();

        if (this.model.get('status') !== TaskModel.STATUS.DRAFT) {
            this.ui.taskHistoryToggler.one('mouseover', () => {
                self.history.currentView.trigger('get:history');
            });
        }
    },

    onClickOut() {
        this.toggleChildrenCnt(false);
    },

    showRegions() {
        if (this.options.selectable) {
            this.selector.close();
            this.selector.show(new TaskSelectorView({
                model: this.model
            }));
        }

        this.imgToggler.close();
        this.imgToggler.show(new TaskImportantTogglerView({
            model: this.model
        }));

        // No need to show history for DRAFT tasks
        if (this.model.get('status') !== TaskModel.STATUS.DRAFT) {
            this.history.close();
            this.history.show(new TaskHistoryView({
                model: this.model
            }));
        }

        this.ownerPriority.close();
        this.ownerPriority.show(new TaskRowControlsView({
            model: this.model
        }));
    },

    toggleChildrenCnt(toggle) {
        if (this.ui.childrenCnt) {
            const button = this.ui.childrenCnt.find('.button_state_check');

            this.ui.childrenCnt.toggleClass(this.options.childrenCntToggleClass, toggle);

            if (!toggle && button) {
                button.removeClass(this.options.childrenCntButtonToggleClass);
            }
        }
    },

    onModelSynced() {
        this.render();
    },

    onModelSyncError() {
        error.loadFail();

        this.model.revertToValidState();
        this.render();
    },

    serializeData() {
        const serialized = BasicTaskLayout.prototype.serializeData.apply(this, arguments);

        serialized.childrenQnt = this.model.getChildrenQnt();
        serialized.childLevel = this.model.getChildLevel();
        serialized.expandable = this.options.expandable;
        serialized.unfilterable = this.options.unfilterable;
        serialized.selectable = this.options.selectable;
        serialized.editable = this.options.editable;
        serialized.isAdmin = this.options.isAdmin;
        serialized.isModal = this.options.isModal;

        return serialized;
    }
});

module.exports = TaskRowView;
