const router = require('../../../../router');

const TaskModel = require('../../../../model/TaskModel');
const TaskRowView = require('../../tasksGrid/row/TaskRowView');
const TaskHistoryView = require('../../tasksGrid/row/TaskHistoryView');
const TaskSelectorView = require('../../tasksGrid/row/TaskSelectorView');
const TaskCompactRowPriorView = require('./TaskCompactRowPriorView');

const clipboardClientMixin = require('../../../components/clipboardClient/ClipboardClientMixin');

/**
 * @class TaskCompactRowView
 * @extends TaskRowView
 *
 * @property {TaskModel} model
 * @property {Marionette.Region} ownerPriority
 * @property {Marionette.Region} priorToggler
 * @property {Marionette.Region} history
 * @property {Marionette.Region} controls
 */
const TaskCompactRowView = TaskRowView.extend({

    template: require('./tpl/TaskCompactRowView.hbs'),

    ui: _.extend({}, TaskRowView.prototype.ui, {
        importantTrigger: '.important_trigger',
        check: '.check',
        button: '.button',
        popup: '.popup',
        popupOverlay: '.popup-overlay',
        tagTrigger: '.tag_trigger'
    }),

    events: _.extend({}, TaskRowView.prototype.events, {
        dblclick: 'onRowSelect',
        'click @ui.importantTrigger': 'onImportantToggle',
        'click @ui.tagTrigger': 'onTagTrigger'
    }),

    behaviors: _.extend({}, TaskRowView.prototype.behaviors, {
        TaskControllerBehavior: {}
    }),

    modelEvents: _.extend({}, TaskRowView.prototype.modelEvents, {
        'change:important': 'render'
    }),

    /** REGIONS DEFINED DIRECTLY IN THE TEMPLATE */

    onRender() {
        this.initBemmyComponents();
        this.showRegions();

        clipboardClientMixin.initializeClipboardClient(
            '.id-proxy-copier',
            'Task ID has been copied to clipboard.'
        );
    },

    initBemmyComponents() {
        this.ui.check.bemmyCheck();
        this.ui.button.bemmyButton();
        this.ui.popupOverlay.bemmyPopupOverlay();
    },

    showRegions() {
        const self = this;

        if (this.options.selectable) {
            this.selector.close();
            this.selector.show(new TaskSelectorView({ model: this.model }));
        }

        this.priorToggler.close();
        this.priorToggler.show(new TaskCompactRowPriorView({ model: this.model }));

        if (this.model.get('status') !== TaskModel.STATUS.DRAFT) {
            this.history.close();
            this.history.show(new TaskHistoryView({
                model: this.model
            }));

            this.ui.taskHistoryToggler.one('mouseover', () => {
                self.history.currentView.$el.removeClass('spinner_hidden');
                self.history.currentView.trigger('get:history');
            });
        }
    },

    onRowSelect() {
        this.model.set('selected', !this.model.get('selected'));

        $('.page').removeClass('page_modal_showed');
    },

    onImportantToggle(evtObj) {
        evtObj.preventDefault();
        this.model.toggleImportant();
    },

    onTagTrigger(evtObj) {
        evtObj.preventDefault();
        router.navigate('/tasks/?tags=' + evtObj.target.innerText.toLowerCase(), { trigger: true });
    }
});

module.exports = TaskCompactRowView;
