""" User authentication """

from __future__ import absolute_import

from sandbox.common import api
from sandbox.common import rest as common_rest
import sandbox.common.types.user as ctu

from .api import Api
from . import schemas


class SSHKey(Api.Path("/authenticate/ssh-key/{login}/{fingerprint}")):
    """ Pseudo-OAuth token generation by SSH key """
    class Post(api.Post):
        """ Pseudo-OAuth token generation by SSH key """
        __content__ = common_rest.Client.BINARY
        __security__ = ctu.Restriction.ANY

        login = api.String(
            "User identifier (login), which should be authenticated",
            required=True, scope=api.Scope.PATH
        )
        fingerprint = api.String(
            "MD5-fingerprint of a key, which is used to authenticate a user",
            required=True, scope=api.Scope.PATH
        )
        signature = api.String(
            "Signature, which was generated by a private key with a fingerprint specified for SHA1 checksum of "
            "user's login, key fingerprint and random number, which was generated by the client",
            scope=api.Scope.BODY
        )
        cnonce = api.String(
            "Hexadecimal representation of a random number, which was generated by a client for the request",
            scope=api.Scope.QUERY
        )

        class Created(api.Response):
            """ Pseudo-OAuth token, which is valid for 3 hours from last usage """
            __content__ = common_rest.Client.PLAINTEXT
            schema = api.String


class OAuthClient(Api.Path("/authenticate/oauth/client")):
    """ OAuth client (application) information """

    class Get(api.Get):
        """ Information required for OAuth token generation request """

        class Ok(api.Response):
            """ Information required for OAuth token generation """
            schema = schemas.authenticate.OAuthClientInfo


class OAuthToken(Api.Path("/authenticate/oauth/token")):
    """ OAuth token manipulation """

    class Post(api.Post):
        """ OAuth token generation """
        __security__ = ctu.Restriction.ANY

        body = schemas.authenticate.OAuthTokenCreate(required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            """ OAuth token produced by an OAuth server based on given code or login/password pair """
            __content__ = common_rest.Client.PLAINTEXT
            schema = api.String

        class BadRequest(api.Response):
            """ OAuth token request failure with the error text """


class ExternalOAuthToken(Api.Path("/authenticate/external/session")):
    """ Create and remove external session tokens """

    class Post(api.Post):
        """ OAuth external token generation """
        __security__ = ctu.Restriction.SESSION_MAKER

        body = schemas.authenticate.ExternalOAuthCreate(required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            """ OAuth external token """
            schema = schemas.authenticate.ExternalOAuth

        class BadRequest(api.Response):
            """ OAuth token request failure with the error text """

        class Forbidden(api.Response):
            """ Forbidden to create oauth external session """

        class Unauthorized(api.Response):
            """ Unauthorized request """

    class Delete(api.Delete):
        """ Delete OAuth external token """
        __security__ = ctu.Restriction.SESSION_MAKER
        body = schemas.authenticate.ExternalOAuthDelete(required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Information required for OAuth token generation """

        class NotFound(api.Response):
            """ Session not found """

        class Forbidden(api.Response):
            """ Forbidden to remove oauth external session """

        class Unauthorized(api.Response):
            """ Unauthorized request """


class ExternalOauthSessionSearch(Api.Path("/authenticate/external/session/search")):
    class Post(api.Post):
        """ OAuth external token getter """
        __security__ = ctu.Restriction.SESSION_MAKER

        body = schemas.authenticate.ExternalOAuthGet(required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            """ External session """
            schema = schemas.authenticate.ExternalOAuth

        class Forbidden(api.Response):
            """ Forbidden to get external session """

        class Unauthorized(api.Response):
            """ Unauthorized request """

        class NotFound(api.Response):
            """ External session not found """

        class BadRequest(api.Response):
            """ Some fields of requests are invalid """


class TaskSessionSearch(Api.Path("/authenticate/task/session/search")):
    class Post(api.Post):
        """ Task session meta information getter """
        __security__ = ctu.Restriction.SESSION_MAKER

        body = schemas.authenticate.TaskSessionGet(required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            """ Task session """
            schema = schemas.authenticate.TaskSession

        class Forbidden(api.Response):
            """ Forbidden to get task session """

        class NotFound(api.Response):
            """ Task session not found """

        class BadRequest(api.Response):
            """ Some fields of requests are invalid """
