""" Operations on client objects """

from __future__ import absolute_import

from sandbox.common import api
from sandbox.common import rest as common_rest
import sandbox.common.types.user as ctu
import sandbox.common.types.client as ctc

from .api import Api
from . import schemas


class ClientList(Api.Path("/client")):
    class Get(api.Get):
        """
        Get client list
        Filter parameters must be passed as query parameters
        """
        id = api.Array(api.String, "Client ID filter")
        platform = api.String("Platform filter")
        alive = api.Boolean("Filter alive clients")
        alive_offset = api.Integer("Apply offset when filtering by alive time", default=0)
        tags = api.String("Client tags query")
        busy = api.Boolean("Filter busy clients")
        search_query = api.String("Filter clients by hostname substring")
        limit = api.Integer("Maximum number of returned entries", required=True)
        offset = api.Integer("Skip this number of entries", default=0)
        fields = api.Array(api.String, "Choose only these fields and client id")

        class Ok(api.Response):
            schema = schemas.client.ClientList

    class Post(api.Post):
        """
        Reset several clients
        """
        body = api.Array(api.String, "Client IDs", scope=api.Scope.BODY)

        class Ok(api.Response):
            schema = api.Array(schemas.client.ResetTask)


class Client(Api.Path("/client/{id}")):
    class Get(api.Get):
        """
        Detailed client information
        """
        id = api.String("Client ID", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.client.ClientEntity

    class Post(api.Post):
        """
        Register launch of new client

        This is an internal method.
        """
        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        body = schemas.client.ClientPing(scope=api.Scope.BODY, required=True)

        class Created(api.Response):
            """New client entry is created"""

        class NoContent(api.Response):
            """Client entry is updated"""

    class Put(api.Put):
        """
        Update client information

        This is an internal method.
        """
        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        body = schemas.client.ClientPing(scope=api.Scope.BODY, required=True)

        class NoContent(api.Response):
            """Client entry is updated"""

        class ResetContent(api.Response):
            schema = schemas.client.ResetReason()
            """Client entry is updated, need current jobs verification"""


class ClientComment(Api.Path("/client/{id}/comment")):
    class Put(api.Put):
        """
        Update client comment
        """
        __content__ = common_rest.Client.PLAINTEXT

        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        comment = api.String(required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            """Operation is successful"""

        class NotFound(api.Response):
            """Client object is not found"""

        class Forbidden(api.Response):
            """Not enough permissions to modify data"""


class ClientTags(Api.Path("/client/{id}/tags")):
    class Put(api.Put):
        """
        Update client tags
        """
        __content__ = common_rest.Client.PLAINTEXT

        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        tags = api.Array(api.String, "Client tag", required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            """Operation is successful"""

        class NotFound(api.Response):
            """Client object is not found"""

        class Forbidden(api.Response):
            """Not enough permissions to modify data"""


class ClientServiceResources(Api.Path("/client/{id}/service/resources")):
    class Get(api.Get):
        """
        Internal interface to get all registered resources on the client
        """
        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        kind = api.Enum("Filters removable client resources", scope=api.Scope.QUERY, values=ctc.RemovableResources)
        limit = api.Integer("Limit amount of resources to be listed", scope=api.Scope.QUERY)

        class Ok(api.Response):
            schema = api.Array(api.Integer, "Resource ID")


class ClientServiceResourcesDrop(Api.Path("/client/{id}/service/resources/drop")):
    __security__ = ctu.Restriction.ADMIN

    class Post(api.Post):
        """
        Internal interface to remove resources on the client
        """
        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        body = api.Array(api.Integer, "Resources id's for drop", required=True, default=[], scope=api.Scope.BODY)

        class NoContent(api.Response):
            """Request has been successfully processed"""


class ClientJob(Api.Path("/client/{id}/job")):
    class Post(api.Post):
        """
        Next task to execute request fetch request

        This is an internal method.
        """
        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        body = schemas.client.ClientGetAJob(scope=api.Scope.BODY, required=True)

        class Ok(api.Response):
            schema = api.Object()

        class Created(api.Response):
            """A new job assigned to client"""
            schema = api.Object()

        class NoContent(api.Response):
            """No job for the client"""

        class ResetContent(api.Response):
            """Client entry is updated, need current jobs verification"""

        class Conflict(api.Response):
            """ Session conflict error """

        class NotFound(api.Response):
            """ Client not found """

        class RequestTimeout(api.Response):
            """ Request timeout """

    class Get(api.Get):
        """
        List of active tasks assigned to client

        This is an internal method.
        """

        id = api.String("Client ID", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.client.ClientJob)

        class NoContent(api.Response):
            """No jobs assigned to client"""

    class Delete(api.Delete):
        """
        Signals the end of task execution

        This is an internal method.
        """
        id = api.String("Client ID", required=True, scope=api.Scope.PATH)

        body = schemas.client.ClientDeleteAJob(scope=api.Scope.BODY, required=True)

        class NoContent(api.Response):
            """Request has been successfully processed"""


class ClientUserTagList(Api.Path("/client/{id}/tags/user")):
    class Post(api.Post):
        """ Add user tag to client """
        __security__ = ctu.Restriction.AUTHENTICATED

        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        user_tag = api.String("User tag name", scope=api.Scope.BODY, required=True)

        class NoContent(api.Response):
            """ Client entry is updated """

        class NotFound(api.Response):
            """ Client, user group or tag not found """

        class Forbidden(api.Response):
            """ User not permitted to use this tag """


class ClientUserTag(Api.Path("/client/{id}/tags/user/{tag}")):
    class Delete(api.Delete):
        """ Remove user tag to client """
        __security__ = ctu.Restriction.AUTHENTICATED

        id = api.String("Client ID", required=True, scope=api.Scope.PATH)
        tag = api.String("User tag name", required=True, scope=api.Scope.PATH)

        class NoContent(api.Response):
            """ Client entry is updated """

        class NotFound(api.Response):
            """ Client, user group or tag not found """

        class Forbidden(api.Response):
            """ User not permitted to use this tag """
