""" Operations on group objects """

from __future__ import absolute_import

from sandbox.common import api
import sandbox.common.types.user as ctu

from .api import Api
from .common import FilterBase
from . import schemas


class GroupList(Api.Path("/group")):
    class Get(FilterBase):
        """
        Get user groups list
        """

        name = api.String("Group name", scope=api.Scope.QUERY)
        parent = api.String("Parent group name", scope=api.Scope.QUERY)
        user = api.String("User name", scope=api.Scope.QUERY)
        user_tag = api.String("User tag", scope=api.Scope.QUERY)
        mds_strong_mode = api.Boolean("Mds strong mode for group", scope=api.Scope.QUERY)
        mds_transfer_resources = api.Boolean("Transfer resources to mds", scope=api.Scope.QUERY)
        fields = api.Array(api.String, "Choose only these fields and group name")
        abc = api.String("Abc group", scope=api.Scope.QUERY)

        class Ok(api.Response):
            """
            Success
            """
            schema = schemas.group.GroupList
            x_matched_records = api.Header(api.Integer("Number of objects matched by the filter"))

    class Post(api.Post):
        """
        Register a new group
        """
        data = schemas.group.GroupCreate(required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            """
            Created
            """
            schema = schemas.group.Group
            location = api.Header("URL of a freshly created group")


class Group(Api.Path("/group/{name}")):
    class Get(api.Get):
        """
        Detailed information about a user group
        """
        __security__ = ctu.Restriction.AUTHENTICATED
        name = api.String("Group name", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            """
            Success
            """
            schema = schemas.group.Group

        class NotFound(api.Response):
            """ Group not found """

    class Put(api.Put):
        """
        Edit group's information
        """
        name = api.String("Group name", required=True, scope=api.Scope.PATH)
        body = schemas.group.GroupUpdate("Data to update", required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            """
            Response will contain updated object if there was an X-Request-Updated-Data header in request
            """
            schema = schemas.group.Group

        class NoContent(api.Response):
            """
            Success
            """

    class Delete(api.Delete):
        """
        Delete a group
        """
        name = api.String("Group name", required=True, scope=api.Scope.PATH)

        class NoContent(api.Response):
            """
            Success
            """


class GroupParent(Api.Path("/group/{name}/parent")):
    class Put(api.Put):
        """
        Add parent for a group
        """
        __security__ = ctu.Restriction.ADMIN

        name = api.String("Group name", required=True, scope=api.Scope.PATH)
        body = schemas.group.GroupParent("Group parent", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            pass
