""" Operations on resource objects """

from __future__ import absolute_import

from sandbox.common import api
import sandbox.common.types.user as ctu
import sandbox.common.types.resource as ctr

from .api import Api
from .common import FilterBase, Ids
from . import schemas


class ResourceList(Api.Path("/resource")):
    class Post(api.Post):
        """Create a NOT_READY resource"""
        body = schemas.resource.ResourceCreate(scope=api.Scope.BODY, required=True)

        class Created(api.Response):
            """Operation is successful"""
            schema = schemas.resource.Resource

        class Forbidden(api.Response):
            """Operation not permitted"""

    class Get(FilterBase):
        """Get a list of resources"""
        id = Ids("Resource Id(s)")
        type = api.Array(api.String, "Filter resource type")
        arch = api.String("Filter architecture type")
        state = api.Array(api.Enum(values=ctr.State), "Filter resource state")
        owner = api.String("Filter resource owner")
        client = api.String("Filter resource client")
        task_id = Ids("Task Id(s)")
        accessed = api.DateTimeRange("Filter by last access time: pair of ISO 8601 dates separated with `..`")
        created = api.DateTimeRange("Filter by creation time: pair of ISO 8601 dates separated with `..`")
        attrs = api.String("""Filter by resource attributes (URL-encoded JSON like `{"param1": "a", "param2": 1}`)""")
        any_attr = api.Boolean("Join attribute filters by logical OR (AND is used by default)")
        attr_name = api.String("Filter by attribute name (deprecated, use `attrs`)")
        attr_value = api.String("Filter by attribute value (deprecated, use `attrs`)")
        dependant = api.Integer("Filter by dependent task ID")

        class Ok(api.Response):
            schema = schemas.resource.ResourceList
            x_matched_records = api.Header(api.Integer("Number of objects matched by the filter"))


class Resource(Api.Path("/resource/{id}")):
    class Get(api.Get):
        """Detailed resource information"""
        id = api.Id(scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.resource.Resource

        class NotFound(api.Response):
            """ Resource not found """

    class Put(api.Put):
        """Change NOT_READY resource or update last access time (touch resource)"""
        id = api.Id(scope=api.Scope.PATH)
        body = schemas.resource.ResourceUpdate(scope=api.Scope.BODY, required=False)

        class NoContent(api.Response):
            """Operation is successful"""

        class Forbidden(api.Response):
            """Operation is not permitted"""

        class NotFound(api.Response):
            """ Resource not found """


class ResourceAttribute(Api.Path("/resource/{id}/attribute")):
    class Get(api.Get):
        """Get resource attributes"""
        id = api.Id(scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.resource.ResourceAttribute)

        class NotFound(api.Response):
            """ Resource not found """

    class Post(api.Post):
        """Create resource attribute"""
        id = api.Id(scope=api.Scope.PATH)
        body = schemas.resource.ResourceAttribute(scope=api.Scope.BODY, required=True)

        class Created(api.Response):
            schema = schemas.resource.ResourceAttribute

        class NotFound(api.Response):
            """ Resource not found """


class ResourceAttributeName(Api.Path("/resource/{id}/attribute/{name}")):
    class Put(api.Put):
        id = api.Id(scope=api.Scope.PATH)
        name = api.String("Attribute name", required=True, scope=api.Scope.PATH)
        body = schemas.resource.ResourceAttributeUpdate(scope=api.Scope.BODY, required=True)

        class NoContent(api.Response):
            """Operation is successful"""

        class NotFound(api.Response):
            """ Resource not found """

    class Delete(api.Delete):
        id = api.Id(scope=api.Scope.PATH)
        name = api.String("Attribute name", required=True, scope=api.Scope.PATH)

        class NoContent(api.Response):
            """Operation is successful"""

        class NotFound(api.Response):
            """ Resource not found """


class ResourceDataRsync(Api.Path("/resource/{id}/data/rsync")):
    class Get(api.Get):
        """List of rsync hosts with URLs"""
        id = api.Id(scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.resource.ResourceUrlItem)

        class NotFound(api.Response):
            """ Resource not found """


class ResourceDataHttp(Api.Path("/resource/{id}/data/http")):
    class Get(api.Get):
        """List of http hosts with URLs"""
        id = api.Id(scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.resource.ResourceUrlItem)

        class NotFound(api.Response):
            """ Resource not found """


class ResourceSource(Api.Path("/resource/{id}/source")):
    class Post(api.Post):
        """Add current session host or given host as a source for a given resource"""
        __security__ = [ctu.Restriction.TASK, ctu.Restriction.ADMIN]

        id = api.Id(scope=api.Scope.PATH)
        body = schemas.resource.ResourceSource(scope=api.Scope.BODY)

        class NoContent(api.Response):
            """Operation is successful"""

        class Forbidden(api.Response):
            """Operation is not permitted"""

        class NotFound(api.Response):
            """ Resource not found """


class ResourceSourceHost(Api.Path("/resource/{id}/source/{host}")):
    class Delete(api.Delete):
        """Remove resource from a host"""
        id = api.Id(scope=api.Scope.PATH)
        host = api.String("Host name", required=True, scope=api.Scope.PATH)

        class NoContent(api.Response):
            """Operation is successful"""

        class NotFound(api.Response):
            """ Resource not found """


class ResourceBackup(Api.Path("/resource/{id}/backup")):
    class Post(api.Post):
        """ Force resource backup """

        id = api.Id(scope=api.Scope.PATH)
        body = schemas.resource.ResourceBackup(scope=api.Scope.BODY)

        class Created(api.Response):
            """ Backup task created """
            schema = schemas.resource.BackupTaskInfo

        class NotFound(api.Response):
            """ Resource not found """


class ResourceInsufficientRedundancy(Api.Path("/resource/{host}/insufficient_redundancy")):
    class Get(api.Get):
        """ List of resources with insufficient redundancy on host """
        host = api.String("Host name", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.resource.ResourceInsufficientRedundancy


class ResourceTouch(Api.Path("/resource/{id}/touch")):
    class Post(api.Post):
        """ Touch resource, don't check, that resource exists"""
        id = api.Id(scope=api.Scope.PATH)

        class NoContent(api.Response):
            """Operation is successful"""


class ResourceMeta(Api.Path("/resource/meta/{rtype}")):
    class Get(api.Get):
        """ Get resource meta information """
        rtype = api.String("Resource type", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.resource.ResourceMeta

        class NotFound(api.Response):
            """ Resource type not found """


class ResourceMetaList(Api.Path("/resource/meta_list/{rtype}")):
    class Get(api.Get):
        """ Get resource meta information """
        rtype = api.String("Resource type", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.resource.ResourceMeta, "Resource meta information")

        class NotFound(api.Response):
            """ Resource type not found """


class ResourceLink(Api.Path("/resource/link")):
    class Get(FilterBase):
        """ Get temporary links for resources for request author"""
        __security__ = [ctu.Restriction.AUTHENTICATED]

        resource_id = api.Id("Filter by resource id")

        class Ok(api.Response):
            schemas = schemas.resource.ResourceLinkList

    class Post(api.Post):
        """ Create temporary link for resource """
        __security__ = [ctu.Restriction.AUTHENTICATED]

        body = schemas.resource.CreateResourceLink(scope=api.Scope.BODY)

        class Ok(api.Response):
            schemas = schemas.resource.ResourceProxyLink

        class NotFound(api.Response):
            """ Resource not found """

    class Put(api.Put):
        """ Update temporary link for resource """
        __security__ = ctu.Restriction.ANY

        body = schemas.resource.UpdateResourceLink(scope=api.Scope.BODY)

        class Ok(api.Response):
            schemas = schemas.resource.ResourceLink

        class NotFound(api.Response):
            """ Link not found """


class ResourcesToBackup(Api.Path("/resource/backup/{client}")):
    class Put(api.Put):
        """ Update current locks and get new resource to backup """
        __security__ = [ctu.Restriction.ADMIN]

        client = api.String(required=True, scope=api.Scope.PATH)
        body = schemas.resource.ActiveBackups(required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            schemas = schemas.resource.ResourcesToBackupResponse

    class Post(api.Post):
        """ Lock resources to upload """
        __security__ = [ctu.Restriction.ADMIN]

        client = api.String(required=True, scope=api.Scope.PATH)
        body = schemas.resource.AcknowledgedBackups(required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            schemas = api.Array(api.Integer, "ids of resources locked to backup")


class LockResourceForMds(Api.Path("/resource/{id}/mds_lock")):
    class Post(api.Post):
        __security__ = [ctu.Restriction.ADMIN]
        id = api.Id(scope=api.Scope.PATH)
        body = schemas.resource.ResourceLockForMds(scope=api.Scope.BODY)

        class Ok(api.Response):
            schemas = schemas.resource.ResourceLockForMdsResponse


class ResourceBuckets(Api.Path("/resource/buckets")):
    class Get(FilterBase):
        """ Get buckets list """
        abc = api.String("ABC service")

        class Ok(api.Response):
            schema = schemas.resource.BucketList

    class Post(api.Post):
        """ Create new bucket (system call) """
        __security__ = [ctu.Restriction.ADMIN]

        body = schemas.resource.Bucket(scope=api.Scope.BODY)

        class Created(api.Response):
            """ Successfully created """
            schemas = schemas.resource.Bucket

        class Forbidden(api.Response):
            """ Operation is not permitted """

        class Conflict(api.Response):
            """ Bucket already exists """

        class BadRequest(api.Response):
            """ Bad request """


class ResourceBucket(Api.Path("/resource/buckets/{name}")):
    class Get(api.Get):
        """ Get bucket """
        name = api.String(required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.resource.Bucket

        class NotFound(api.Response):
            """ Bucket is not found by name """

    class Put(api.Put):
        """ Update bucket """
        __security__ = [ctu.Restriction.AUTHENTICATED]

        name = api.String(required=True, scope=api.Scope.PATH)
        body = schemas.resource.Bucket(scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Operation is successful """

        class Forbidden(api.Response):
            """ User has no permissions to edit the bucket """

        class NotFound(api.Response):
            """ Bucket is not found by name """
