""" Operations on scheduler objects """

from __future__ import absolute_import

from sandbox.common import api

from .api import Api
from .common import FilterBase, Ids
from . import schemas


class SchedulerList(Api.Path("/scheduler")):
    class Get(FilterBase):
        """
        Get list of schedulers
        """
        task_type = api.Array(api.String, "Task type")
        id = Ids("Scheduler identifier")
        status = api.Array(api.String, "Scheduler status")
        owner = api.String("Scheduler owner")
        author = api.String("Scheduler author (the creator)")
        tags = api.Array(api.String, "Task tags")
        all_tags = api.Boolean("Accept tasks with if it has all the tags, any of the tags otherwise")

        class Ok(api.Response):
            """
            Success
            """
            schema = schemas.scheduler.SchedulerList
            x_matched_records = api.Header(api.Integer("Number of objects matched by the filter"))

    class Post(api.Request):
        """
        Create a scheduler
        """
        data = schemas.scheduler.SchedulerNew("Data to create a scheduler from", scope=api.Scope.BODY)

        class Created(api.Response):
            """
            Created
            """
            schema = schemas.scheduler.Scheduler
            location = api.Header("URL of a freshly created scheduler")


class Scheduler(Api.Path("/scheduler/{id}")):
    class Get(api.Request):
        """
        Detailed information about a scheduler
        """
        id = api.Id("Global identifier", scope=api.Scope.PATH)

        class Ok(api.Response):
            """
            Success
            """
            schema = schemas.scheduler.Scheduler

    class Put(api.Request):
        """
        Update information about a scheduler
        On success return updated object if there was header X-Request-Updated-Data
        """
        id = api.Id("Scheduler identifier", scope=api.Scope.PATH)
        body = schemas.scheduler.SchedulerUpdate("Parameters to update", required=True, scope=api.Scope.BODY)

        class BadRequest(api.Response):
            """
            Bad request (you don't say)
            """

        class NotFound(api.Response):
            """
            Scheduler not found
            """

        class NoContent(api.Response):
            """
            Successfully updated a scheduler
            """

        class Ok(api.Response):
            """
            Response will contain updated object if there was an X-Request-Updated-Data header in request
            """
            schema = schemas.scheduler.Scheduler

    class Delete(api.Request):
        """ Delete the given scheduler """
        id = api.Id("Scheduler identifier", scope=api.Scope.PATH)

        class NoContent(api.Response):
            """ Scheduler has been deleted successfully """

        class NotFound(api.Response):
            """ No scheduler with ID provided can be found """

        class Forbidden(api.Response):
            """ Operation is not permitted """


class SchedulerFields(Api.Path("/scheduler/{id}/custom/fields")):
    class Get(api.Request):
        """
        Custom fields of tasks created by a scheduler
        Vital for a scheduler view/edit form
        """
        id = api.Id("Scheduler identifier", scope=api.Scope.PATH)

        class Ok(api.Response):
            """
            Success
            """
            schema = api.Array(schemas.task.TaskCustomFieldMeta)

    class Post(api.Request):
        """
        Validate task's custom fields
        """
        id = api.Id("Scheduler identifier", scope=api.Scope.PATH)
        body = api.Array(schemas.task.TaskFieldValidateItem, required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            """
            Success
            """
            schema = api.Array(schemas.task.TaskFieldValidateResultItem)

    class Put(api.Request):
        """
        Update task's custom fields
        """
        id = api.Id("Scheduler identifier", scope=api.Scope.PATH)
        body = api.Array(
            schemas.task.TaskFieldValidateItem, "Parameters to update", required=True, scope=api.Scope.BODY
        )

        class Ok(api.Response):
            """
            Success
            """
            schema = api.Array(schemas.task.TaskFieldValidateResultItem)
