from __future__ import absolute_import

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

from sandbox.common import api

from ..api import Api

from . import common


class DiskInfo(Api.Schema):
    """
    Disk info for client
    """
    status = api.Enum("Disk status", values=ctc.DiskStatus, required=True)
    total_space = api.Integer(required=True, format=api.IntegerFormat.I64)
    free_space = api.Integer(required=True, format=api.IntegerFormat.I64, minimum=None)
    locked_space = api.Integer(format=api.IntegerFormat.I64)


class ClientOS(Api.Schema):
    """
    Information about client's OS
    """
    name = api.String("OS distribution")
    version = api.String("OS version")


class ClientTaskOne(Api.Schema):
    """
    Information about currently executing task on client
    """
    task_id = api.Id("Task ID", required=True)
    task_type = api.String("Task type", required=True)
    owner = api.String("Task owner", required=True)
    url = api.String("Task URL", required=True)


class ClientTask(Api.Schema):
    """
    Information about currently executing task on client
    """
    id = api.Id("Task ID", required=True)
    type = api.String("Task type", required=True)
    owner = api.String("Task owner", required=True)
    url = api.String("Task URL", required=True)


class UserTags(Api.Schema):
    allowed = api.Map(api.Array(api.String, "User tags"), "User tags allowed to modify by request author")
    others = api.Map(api.Array(api.String, "User tags"), "User tags not allowed to modify by request author")


class ClientEntity(Api.Schema):
    """
    Client information
    """
    id = api.String("Client identifier")
    uuid = api.String("Client uuid")
    url = api.String("URL for client information")
    os = ClientOS()
    cpu = api.String("CPU model")
    ncpu = api.Integer("Number of CPU cores")
    ram = api.Integer("RAM size in bytes", format=api.IntegerFormat.I64)
    platform = api.String("Client platform identifier")
    platforms = api.Array(api.String, "Client container's platform")
    lxc = api.Boolean("Whether LXC containers are enabled")
    fqdn = api.String("Client's FQDN")
    fileserver = api.String("Root URL of client's fileserver")
    dc = api.String("3-letter ID of client's datacenter")
    task = ClientTaskOne("Any task currently executing on client")
    tasks = api.Array(ClientTask, "Tasks currently executing on client")
    alive = api.Boolean("Whether client is alive")
    availability = api.Enum("Availability of client", values=ctc.ClientAvailability)
    disk = DiskInfo()
    tags = api.Array(api.String, "Client tag")
    # age = api.Integer("Protocol version")
    last_activity = api.DateTime("Time of last API call")
    pending_commands = api.Array(api.String, "Pending service command(s)")
    net_unreachable = api.DateTime("If unreachable, time of last check that detected unreachability")
    msg = api.String("Client comment/description")
    owners = api.Array(api.String, "Client owners")
    user_tags = UserTags("user tags")


class ClientJob(Api.Schema):
    """
    Executing job information
    """
    id = api.String("Task session token", required=True)
    state = api.Enum("Job action", values=ctt.SessionState, required=True)
    options = api.Object("Job options")


class ComputingResources(Api.Schema):
    """
    Computing resources
    """
    ram = api.Integer("RAM, in MiB", required=True)
    cores = api.Integer("CPU cores", required=True)


class ClientGetAJob(Api.Schema):
    """
    Client job request
    """
    age = api.Integer("Client protocol version", required=True)
    disk_free = api.Integer("Available disk space for the task execution in bytes", required=True, minimum=None)
    tokens = api.Array(api.String, "Task sessions tokens", scope=api.Scope.BODY)
    free = ComputingResources("Free computing resources")


class ClientDeleteAJob(Api.Schema):
    """
    Client job request
    """
    token = api.String("Task session token", required=True, scope=api.Scope.BODY)
    reject = api.Boolean("Whether client has rejected the job", scope=api.Scope.BODY)
    reject_type = api.Enum("Enum for rejection reason", values=ctc.RejectionReason, scope=api.Scope.BODY)
    reason = api.String("Reason for the job rejection if there is any", scope=api.Scope.BODY)
    restart = api.Boolean("Job rejected because of infrastructure problem and may be restarted", scope=api.Scope.BODY)
    target_status = api.Enum("Current task status, use to prevent conflict", values=ctt.Status)
    wait_targets = api.Object("Dictionary of time, tasks or resources awaited by the task")


class ClientPingHardwareCPU(Api.Schema):
    """
    CPU information
    """
    cores = api.Integer("Number of cores", required=True)
    model = api.String("ID of CPU model", required=True)
    vendor = api.Enum("CPU manufacturer", values=ctm.CpuManufacturer, required=True)
    frequency = api.Integer("Nominal frequency in GHz", required=True)
    architecture = api.Enum("CPU architecture", values=ctm.CpuArchitecture, required=True)


class ClientPingHardwareRAM(Api.Schema):
    """
    RAM information
    """
    total = api.Integer("Total RAM in bytes", required=True, format=api.IntegerFormat.I64)
    free = api.Integer("Free RAM in bytes", required=True, format=api.IntegerFormat.I64)


class ClientPingHardwareDisk(Api.Schema):
    """
    Disk subsystem information
    """
    type = api.Enum("Disk type", values=ctm.StorageDeviceType, required=True)
    total = api.Integer("Total volume dedicated to task execution", required=True, format=api.IntegerFormat.I64)
    free = api.Integer("Free volume for task execution", required=True, format=api.IntegerFormat.I64)


class ClientPingHardwareNetwork(Api.Schema):
    """
    Network interface information
    """
    fastbone = api.Boolean("Is Fastbone enabled?", required=True)
    ipv4 = api.Boolean("Is IPv4 available?", required=True)
    ipv6 = api.Boolean("Is IPv6 available?", required=True)
    fqdn = api.String("Fully qualified domain name", required=True)


class ClientPingHardware(Api.Schema):
    """
    Client system information
    """
    cpu = ClientPingHardwareCPU(required=True)
    ram = ClientPingHardwareRAM(required=True)
    disk = ClientPingHardwareDisk(required=True)
    network = ClientPingHardwareNetwork(required=True)


class ClientSlots(Api.Schema):
    used = api.Integer("Number of used slots", required=True)
    total = api.Integer("Number of total slots", required=True)


class ResetReason(Api.Schema):
    reason = api.String("Reset reason")


class ClientPing(Api.Schema):
    """
    Client information update
    """
    uuid = api.String("Client uuid")
    arch = api.String("OS distribution")
    os = ClientOS(required=True)
    disk = DiskInfo(required=True)
    slots = ClientSlots()
    ram = api.Integer("RAM size in bytes", format=api.IntegerFormat.I64, required=True)
    cpu = api.String("CPU model", required=True)
    ncpu = api.Integer("Number of CPU cores", required=True)
    platform = api.String("Client platform identifier", required=True)
    lxc = api.Boolean("Whether LXC containers are enabled")
    porto = api.Boolean("Whether porto is enabled")
    root = api.Boolean("Root privileges", required=True)
    fqdn = api.String("Client's FQDN")
    dc = api.String("3-letter ID of client's datacenter", required=True)
    fileserver = api.String("Root URL of client's fileserver", required=True)
    tasks_dir = api.String("Task directory", required=True)
    tags = api.Array(api.String, "Client tag")

    sdk = api.String("Sdk revision")
    client = api.String("Client revision")
    tasks = api.String("Tasks revision")
    venv = api.String("Venv revision")
    server = api.String("Server revision")

    age = api.Integer("Protocol version")
    jobs = api.Array(ClientJob, "Currently executing task", default=[])


class ClientList(common.List):
    """
    Client list
    """
    item = ClientEntity


class ResetTask(Api.Schema):
    """
    Information about task reset on client
    """
    id = api.Id("Task ID", required=True)
    host = api.String("Task host", required=True)
