from __future__ import absolute_import

import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
from sandbox.common import api

from ..api import Api

from . import abcd
from . import task
from . import user
from . import common


class GroupPriorityLimits(Api.Schema):
    """
    Maximum allowed task priority for this group
    """
    api = task.TaskPriority(required=True)
    ui = task.TaskPriority(required=True)


class GroupSyncSourceContent(Api.Schema):
    name = api.String("Group name")
    logins = api.Array(api.String, "Group actual users")


class GroupSyncSource(Api.Schema):
    """
    External source to synchronize user list with
    """
    source = api.Enum("Source type", values=ctu.GroupSource)
    group = api.String("Groups names in external source")
    content = api.Array(GroupSyncSourceContent, "Groups content")


class JugglerCheck(Api.Schema):
    host = api.String("Check host")
    service = api.String("Check service")
    namespace = api.String("Check namespace. If value not set, use default namespace")


class JugglerSettings(Api.Schema):
    default_host = api.String("Default check host")
    default_service = api.String("Default check service")
    checks = api.Map(JugglerCheck, "Juggler checks")


class GroupMDSQuota(Api.Schema):
    """
    Quota and consumption of the group in MDS/S3
    """
    name = api.String("Bucket name")
    used_space = api.Integer("Used space in bucket, in bytes")
    max_size = api.Integer("Max bucket size, in bytes")


class Group(Api.Schema):
    """
    Information about user groups
    """
    url = api.String("URL to get detailed information from", required=True)
    abc = api.String("ABC service identifier the group is linked with")
    name = api.String("Group name", required=True)
    rights = api.Enum(
        "Current user's access rights for this resource",
        required=True,
        values=ctu.Rights
    )
    members = api.Array(api.String, "Users names", required=True)
    users = api.Map(api.Array(api.String, "User logins"), "Users on sources")
    parent = api.String("Parent group name")
    email = api.String("E-mail address")
    priority_limits = GroupPriorityLimits(required=True)
    sync = GroupSyncSource()
    sources = api.Array(GroupSyncSource, "Group sources")
    quota = user.QuotaInfo()
    mds_quota = GroupMDSQuota()
    user_tags = api.Array(api.String, "User tags")
    messenger_chat_id = api.String("Group chat id in messenger Q")
    telegram_chat_id = api.String("Telegram chat id")
    juggler_settings = JugglerSettings("Juggler settings")
    mds_strong_mode = api.Boolean("Mds strong mode for group")
    mds_transfer_resources = api.Boolean("Transfer resources to mds")
    abcd_account = abcd.Account()


class GroupCreate(Api.Schema):
    """
    User group registration
    """
    name = api.String("Group name", required=True)
    email = api.String("E-mail address")
    sources = api.Array(GroupSyncSource, "Sync sources")
    abc = api.String("ABC service")
    messenger_chat_id = api.String("Group chat id in messenger Q")
    telegram_chat_id = api.String("Telegram chat id")
    juggler_settings = JugglerSettings("Juggler settings")
    user_tags = api.Array(api.String, "Client tags of group")
    mds_strong_mode = api.Boolean("Mds strong mode for group")
    mds_transfer_resources = api.Boolean("Transfer resources to mds")


class GroupUpdate(Api.Schema):
    """
    Change group's information
    """
    name = api.String("Group name", default=ctm.NotExists)
    email = api.String("E-mail address", default=ctm.NotExists)
    sources = api.Array(GroupSyncSource, "Sync sources", default=ctm.NotExists)
    abc = api.String("ABC service", default=ctm.NotExists)
    messenger_chat_id = api.String("Group chat id in messenger Q", default=ctm.NotExists)
    telegram_chat_id = api.String("Telegram chat id")
    juggler_settings = JugglerSettings("Juggler settings")
    user_tags = api.Array(api.String, "Client tags of group")
    mds_strong_mode = api.Boolean("Mds strong mode for group")
    mds_transfer_resources = api.Boolean("Transfer resources to mds")


class GroupList(common.List):
    """
    List of groups
    """
    item = Group


class GroupParent(Api.Schema):
    parent = api.String("Parent group name")
