from __future__ import absolute_import

import sandbox.common.types.user as ctu
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
from sandbox.common import api

from ..api import Api

from . import abcd
from . import common


class ResourceTime(Api.Schema):
    """Base resource timestamps"""
    created = api.DateTime("Resource creation time", required=True)
    accessed = api.DateTime("Last access time", required=True)
    expires = api.DateTime("Approximate time of resource expiration")
    updated = api.DateTime("Last update time", required=True)


class ResourceDataLinks(Api.Schema):
    links = api.Array(api.String, "Link to resource")
    proxy = api.String("Link to proxy")


class ResourceTask(Api.Schema):
    id = api.Id()
    url = api.String("URI to receive data", required=True)
    status = api.String("Task status", required=True)


class MDSUpdate(Api.Schema):
    """ Update metadata of data saved in MDS """
    key = api.String("MDS key", required=True)
    namespace = api.String("MDS namespace", default=ctr.DEFAULT_S3_BUCKET)


class MDS(MDSUpdate):
    """ Metadata of data saved in MDS """
    url = api.String("S3 url")
    backup_url = api.String("Backup S3 url")


class SystemAttributes(Api.Schema):
    linux_platform = api.String("Path of binary for linux platform")
    osx_platform = api.String("Path of binary for osx platform")
    osx_arm_platform = api.String("Path of binary for osx_arm platform")
    win_nt_platform = api.String("Path of binary for win_nt platform")


class AttributesMeta(Api.Schema):
    class_name = api.String("Attribute type")
    description = api.String("Attribute description")
    default = api.DataType("Attribute default")
    required = api.Boolean("Attribute is required")
    value = api.DataType("Attribue value")


class ResourceMeta(Api.Schema):
    name = api.String("Resource type", required=True)
    parent = api.String("Resource class sdk parent", required=True)
    releaseable = api.Boolean("Resource is releaseable")
    releasable = api.Boolean("Resource is releasable")
    any_arch = api.Boolean("Accept any arch")
    releasers = api.Array(api.String, "List of resource releasers")
    executable = api.Boolean("Executable")
    auto_backup = api.Boolean("Resource must be backuped")
    calc_md5 = api.Boolean("Calc md5 for resource")
    on_restart = api.String("On restart policy")
    release_subscribers = api.Array(api.String, "Release subscribers")
    share = api.Boolean("Resource must be share by skynet")
    default_attribute = api.String("Hack field for AbstractResource")
    attributes = api.Map(
        api.Object, "Attributes name -> value with value keys: class_name, description, default, required, value"
    )


class ResourceBase(Api.Schema):
    """Base class for resources"""
    id = api.Id()
    rights = api.Enum("Resource permissions for current user", values=ctu.Rights, required=True)
    url = api.String("Resource info URL", required=True)
    task = ResourceTask("Task that created resource", required=True)
    http = ResourceDataLinks("HTTP links to resource (may be missing for unauthorized users)")
    arch = api.String("Architecture identifier", required=True)
    state = api.Enum("Current state", values=ctr.State, required=True)
    type = api.String("Type of resource", required=True)
    description = api.String("Resource description", required=True)
    owner = api.String("Resource owner", required=True)
    time = ResourceTime(required=True)
    attributes = api.Object("Resource attributes", required=True)
    size = api.Integer("Resource size in bytes", required=True)
    md5 = api.String("MD5 hash of resource")
    skynet_id = api.String("Skynet ID")
    file_name = api.String("Relative path to resource file", required=True)
    mds = MDS()
    multifile = api.Boolean("Multifile resource", default=False)
    executable = api.Boolean("Executable bit", default=False)
    observable = api.Boolean("Resource can be observed", default=False)
    resource_meta = api.Array(ResourceMeta, "Resource meta information")
    system_attributes = SystemAttributes("Resource system attributes")


class ResourseInsufficientRedundancyItem(Api.Schema):
    """ Resource with insufficient redundancy information """
    id = api.Id()
    type = api.String("Type of resource", required=True)
    size = api.Integer("Resource size in bytes", required=True)


class ResourceListItem(ResourceBase):
    """Resource information"""


class ResourceUrlItem(Api.Schema):
    """Resource data URL"""
    url = api.String("URL", required=True)
    host = api.String("Host name", required=True)
    storage = api.Boolean("Is the host a storage?", required=True)
    alive = api.Boolean("Is the host alive?", required=True)
    state = api.Enum("Resource host state", values=ctr.HostState, required=True)


class ResourceSource(Api.Schema):
    """Resource source info"""
    host = api.String("Host name")
    mds = MDS()


class Resource(ResourceBase):
    sources = api.Array(api.String, "Clients IDs", required=True)
    rsync = ResourceDataLinks("List of rsync links")
    for_parent = api.Boolean("Create resource for parent of current task")


class ResourceCreate(Api.Schema):
    """"Information for resource create"""
    arch = api.String("Architecture identifier")
    state = api.Enum("Current state", values=ctr.State)
    type = api.String("Type of resource", required=True)
    description = api.String("Resource description")
    time = ResourceTime()
    attributes = api.Object("Resource attributes")
    size = api.Integer("Resource size in bytes")
    md5 = api.String("MD5 hash of resource")
    skynet_id = api.String("Skynet ID")
    file_name = api.String("Relative path to resource file")
    mds = MDSUpdate()
    multifile = api.Boolean("Multifile resource", default=False)
    executable = api.Boolean("Executable bit", default=False)
    for_parent = api.Boolean("Created for parent task", default=False)
    resource_meta = api.Array(ResourceMeta, "Resource meta information")
    system_attributes = SystemAttributes("Resource system attributes")


class ResourceUpdate(Api.Schema):
    """Information for resource update"""
    arch = api.String("Architecture identifier", default=ctm.NotExists)
    state = api.Enum("Current state", values=ctr.State, default=ctm.NotExists)
    description = api.String("Resource description", default=ctm.NotExists)
    attributes = api.Object("Resource attributes", default=ctm.NotExists)
    size = api.Integer("Resource size in bytes", default=ctm.NotExists)
    md5 = api.String("MD5 hash of resource", default=ctm.NotExists)
    skynet_id = api.String("Skynet ID", default=ctm.NotExists)
    file_name = api.String("Relative path to resource file", default=ctm.NotExists)
    mds = MDSUpdate(default=ctm.NotExists)
    multifile = api.Boolean("Multifile resource", default=ctm.NotExists)
    executable = api.Boolean("Executable bit", default=ctm.NotExists)


class ResourceAttribute(Api.Schema):
    name = api.String("Attribute name", required=True)
    value = api.String("Attribute value", required=True)


class ResourceBackup(Api.Schema):
    """ Resource backup request optional information """
    dc = api.String("Resource source DC identifier")
    size = api.Integer("Resource size in bytes")


class BackupTaskInfo(Api.Schema):
    """ Resource backup task information """
    id = api.Id()


class ResourceList(common.List):
    item = ResourceListItem


class ResourceInsufficientRedundancy(common.List):
    """ List of resource items with insufficient redundancy on host """
    item = ResourseInsufficientRedundancyItem


class ResourceAttributeUpdate(Api.Schema):
    name = api.String("Attribute name", default=ctm.NotExists)
    value = api.String("Attribute value", default=ctm.NotExists)


class CreateResourceLink(Api.Schema):
    resource_id = api.Id("Resource id", required=True)


class UpdateResourceLink(Api.Schema):
    id = api.String("Uuid for resource and author", required=True)


class ResourceLink(Api.Schema):
    id = api.String("Uuid for resource and author", required=True)
    resource_id = api.Id("Resource id", required=True)
    author = api.String("Link author", required=True)
    accessed = api.DateTime("Last usage of resource link", required=True)


class ResourceProxyLink(ResourceLink):
    link = api.String("Temporary proxy resource link")


class ResourceLinkList(common.List):
    item = ResourceProxyLink


class ActiveBackups(Api.Schema):
    resources = api.Array(api.Integer, "Ids of resources currently backing up by host")


class ResourcesToBackupResponse(Api.Schema):
    resources = api.Array(api.Integer, "Ids of resources need to backup")
    cancelled = api.Array(api.Integer, "Ids of resources not locked to the client")


class AcknowledgedBackups(Api.Schema):
    resources = api.Array(api.Integer, "Ids of resources acknowledged to backup by host")


class ResourceLockForMds(Api.Schema):
    host = api.String("Host for lock", required=True)
    acquire = api.Boolean("Acquire lock if True and release if False", required=True)


class ResourceLockForMdsResponse(Api.Schema):
    result = api.Boolean("Acquiring or releasing result", required=True)


class Bucket(Api.Schema):
    name = api.String("Name")
    abc = api.String("ABC service")
    abcd_account = abcd.Account()
    ignore_bucket_exhaust = api.Boolean("Ignore bucket exhausting")
    lru_threshold = api.Integer("LRU threshold, bytes")


class BucketList(common.List):
    item = Bucket
