from __future__ import absolute_import

from sandbox.common import api
import sandbox.common.types.user as ctu
import sandbox.common.types.scheduler as cts

from ..api import Api

from . import task
from . import common


class SchedulerLastTask(Api.Schema):
    """
    Information about the latest task created
    """
    id = api.Id("Global identifier")
    url = api.String("URL to get detailed information from", required=True)
    status = api.String("Current status")


class BaseTaskConfig(Api.Schema):
    """
    Information about a task created by the scheduler
    """
    last = SchedulerLastTask("The latest task")
    type = api.String("Type of tasks being created", required=True)
    description = api.String("Task description", required=True)
    tags = api.Array(api.String, "Task tags", required=True)


class SchedulerTime(Api.Schema):
    """
    Time settings
    """
    created = api.DateTime("Creation time (ISO 8601 format)", required=True)
    updated = api.DateTime("Update time (ISO 8601 format)", required=True)
    next = api.DateTime("Next start time")
    last = api.DateTime("Last start time")


class Repetition(Api.Schema):
    """
    Contains either `interval` or `weekly` field
    """
    interval = api.Integer("Interval to start tasks with, in seconds")
    weekly = api.Array(api.Integer(maximum=6), "Day of a week to start a task on; 0 is Monday, ..., 6 is Sunday")


class Retry(Api.Schema):
    """
    Settings for restarting a task after the latest one has failed
    """
    ignore = api.Boolean("Ignore fails")
    interval = api.Integer("Interval to restart a task within, in seconds")


class SchedulerNotifications(task.BaseNotifications):
    statuses = api.Array(
        api.Enum(values=cts.Status), "Scheduler statuses when notifications should be sent", required=True
    )


class SchedulerInterval(Api.Schema):
    """
    Interval settings
    """
    start_time = api.DateTime("Start time (ISO 8601 format). If not specified, the scheduler will start immediately")
    repetition = Repetition("Interval to start tasks within")
    retry = Retry("Interval to start tasks within, in case the latest task has failed")
    fail_on_error = api.Boolean("Set status to FAILURE in case of task failure")
    sequential_run = api.Boolean("Start a new task only after the previous one's finish")


class SchedulerListItem(Api.Schema):
    """
    Schedulers list item
    """
    id = api.Id("Global identifier")
    url = api.String("URL to get detailed information from", required=True)
    rights = api.Enum(
        "Current user's access rights for this resource",
        required=True,
        values=ctu.Rights
    )
    status = api.Enum("Current status", required=True, values=cts.Status)
    task = BaseTaskConfig("Properties of tasks to create")
    owner = api.String("Owner", required=True)
    author = api.String("Author (the creator)", required=True)
    scheduler_notifications = api.Array(SchedulerNotifications, "Scheduler notification rule")
    time = SchedulerTime("Time settings", required=True)
    schedule = SchedulerInterval("Settings of interval to (re)start tasks within", required=True)


class TaskConfig(BaseTaskConfig):
    """
    Tasks' common settings
    """
    sdk_version = api.Integer("Task's SDK version")
    owner = api.String("Owner", required=True)
    priority = task.TaskPriority(required=True)
    requirements = task.TaskRequirements(required=True)
    important = api.Boolean("Whether a task is important")
    kill_timeout = api.Integer("Maximum allowed time to execute, in seconds", required=True)
    fail_on_any_error = api.Boolean("Switch task's state to FAILURE on any error", required=True)
    tasks_archive_resource = api.Id("Task code resource Id of type SANDBOX_TASKS_(ARCHIVE|BINARY)", required=False)
    hidden = api.Boolean("Whether a task is hidden", required=True)
    notifications = api.Array(task.TaskNotifications, "Notification rule")
    custom_fields = api.Array(
        task.TaskFieldValidateItem,
        "User defined task fields (shortcut to PUT request for '/scheduler/{id}/custom/fields' path)"
    )


class Scheduler(Api.Schema):
    """
    Information about scheduler
    """
    id = api.Id("Global identifier")
    url = api.String("URL to get detailed information from", required=True)
    rights = api.Enum(
        "Current user's access rights for this resource",
        required=True,
        values=ctu.Rights
    )
    owner = api.String("Owner")
    author = api.String("Author (the creator)", required=True)
    scheduler_notifications = api.Array(SchedulerNotifications, "Scheduler notification rule")
    time = SchedulerTime("Time settings", required=True)
    schedule = SchedulerInterval("Settings of interval to (re)start tasks within", required=True)
    task = TaskConfig("Tasks' common settings", required=True)


class SchedulerUpdate(Api.Schema):
    """
    Information about a task being updated
    """
    owner = api.String("Owner of the scheduler and its children")
    author = api.String("Author of the scheduler and its children; can only be changed to an API request author")
    scheduler_notifications = api.Array(SchedulerNotifications, "Scheduler notification rule")
    schedule = SchedulerInterval("Settings of interval to (re)start tasks within")
    task = task.TaskUpdate(
        "Information about task settings being updated; owner is inherited from a scheduler"
    )


class SchedulerNew(Api.Schema):
    """
    Data to create a new scheduler from (pick one)
    """
    task_type = api.String("Task type")
    source = api.Id("Identifier of a scheduler to clone", required=False)
    data = SchedulerUpdate("Initial data", required=False)


class SchedulerList(common.List):
    item = SchedulerListItem
