from __future__ import absolute_import

import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
from sandbox.common import api

from ..api import Api

from . import common


class SemaphoreTime(Api.Schema):
    """ Time attributes of semaphore """
    created = api.DateTime("Semaphore creation time, UTC, ISO 8601", required=True)
    updated = api.DateTime("Semaphore update time, UTC, ISO 8601", required=True)


class SemaphoreTask(Api.Schema):
    """ Contains identifier and weight of the task acquired the semaphore """
    task_id = api.Id("Task id")
    weight = api.Integer("Task weight", required=True, minimum=1)


class SemaphoreBase(Api.Schema):
    name = api.String("Semaphore name", required=True)
    owner = api.String("Group that allowed to modify the semaphore", required=True)
    shared = api.Array(api.String, "List of groups that allowed to use the semaphore from tasks")


class SemaphoreListItem(SemaphoreBase):
    id = api.Id("Semaphore id")
    url = api.String("URL to fetch the semaphore", required=True)
    rights = api.Enum("Allowed rights on the semaphore for the current user", values=ctu.Rights)
    time = SemaphoreTime("Time marks", required=True)
    capacity = api.Integer("Maximum capacity", required=True, minimum=0)
    value = api.Integer("Current value", required=True)
    auto = api.Boolean("Created automatically")
    public = api.Boolean("Public semaphore")


class Semaphore(SemaphoreListItem):
    """ Semaphore information """

    tasks = api.Array(SemaphoreTask, "Tasks acquired the semaphore")
    statistics_url = api.String("URL to Solomon graphic")


class SemaphoreCreate(SemaphoreBase):
    """ New semaphore creation data """
    capacity = Semaphore.capacity(required=False)
    public = api.Boolean("Public semaphore")


class SemaphoreUpdate(Api.Schema):
    """ Update data for existing semaphore """
    owner = api.String("Group that allowed to modify the semaphore", default=ctm.NotExists)
    shared = api.Array(api.String, "List of groups that allowed to use the semaphore from tasks", default=ctm.NotExists)
    capacity = api.Integer("Maximum capacity", minimum=0, default=ctm.NotExists)
    auto = api.Boolean("Created automatically", default=ctm.NotExists)
    public = api.Boolean("Public semaphore", default=ctm.NotExists)
    event = api.String("Update description")


class SemaphoreList(common.List):
    item = SemaphoreListItem
    groups = api.Array(api.String, "Groups")


class SemaphoreAuditItem(Api.Schema):
    """ Semaphore audit item """
    time = api.DateTime("Update time, UTC, ISO 8601", required=True)
    description = api.String("Description", required=True)
    target = api.String("Host that handled update", required=True)
    author = api.String("Update initiator")
    source = api.String("Network address of update initiator")
