from __future__ import absolute_import

import sandbox.common.types.misc as ctm
import sandbox.common.types.notification as ctn
from sandbox.common import api

from ..api import Api

from . import task
from . import common


class ServiceThreadStatusTime(Api.Schema):
    """ Information about times of the service thread """
    last = api.DateTime("Time of last start", required=True)
    next_ = api.DateTime("Time of next start", required=True)


class ServiceThreadStatus(Api.Schema):
    """ Information about status of service thread """
    id = api.String("Service thread name", required=True)
    run_time = ServiceThreadStatusTime(required=True)
    locks = api.Array(api.String, "Information about locks (first in the list ownes the lock)", required=True)


class DatabaseShardReplicaInstance(Api.Schema):
    """ Information about process `mongod` of replica set of the shard """
    id = api.String("Id of the process `mongod` in format <hostname>:<port>", required=True)
    uptime = api.Integer("Working time for the process in seconds from start", format=api.IntegerFormat.I64)
    optime = api.String("Id of the last synced record of operation log in format <timestamp>.<operation_number>")
    ping = api.Integer(
        "Duration of the last command 'ping' for the server in milliseconds",
        required=True, format=api.IntegerFormat.I64
    )
    sync_source = api.String("Id of the process `mongod` in format <hostname>:<port> from which syncing data")
    last_heartbeat = api.DateTime("Time (UTC) of the last info packet of the process")
    state = api.Enum("Process state", values=ctm.MongoState)
    elected = api.String("Time (UTC) of the process elected as PRIMARY")
    can_vote = api.Boolean("Does this shared takes part in quorum confirmation")
    hidden = api.Boolean("Shard is hidden and doesn't receive any traffic except basic replication")
    shard_cpu_usage = api.Number("Cpu usage of shard")
    shard_reloaded_ts = api.DateTime("Timestamp of the last shard restart")
    shard_cpu_updated = api.DateTime("Timestamp of the last shard_cpu_usage update")


class DatabaseShard(Api.Schema):
    """ Information about shards of MongoDB """
    id = api.String("Shard Id", required=True)
    updated = api.DateTime("Last update time of the shard", required=True)
    replicaset = api.Array(DatabaseShardReplicaInstance, required=True)


class DatabaseSize(Api.Schema):
    """ Information about DB size """
    indexsize = api.Integer("Indexes size in bytes", format=api.IntegerFormat.I64)
    datasize = api.Integer("Data size in bytes", format=api.IntegerFormat.I64)
    filesize = api.Integer("Files size in bytes", format=api.IntegerFormat.I64)
    avgobjsize = api.Number("Average object size in bytes")


class DatabaseCurrentOp(Api.Schema):
    """ Information about duration of requests to DB """
    duration = api.Number("")
    read = api.Integer("", format=api.IntegerFormat.I64)
    current_op = api.Integer("", format=api.IntegerFormat.I64)
    read_lock = api.Number("")
    write = api.Integer("", format=api.IntegerFormat.I64)
    write_lock = api.Number("")


class WebServerStatusRequestsCount(Api.Schema):
    """ Counters of processed requests """
    low_priority = api.Integer("Number of high priority (UI) requests processed from the server start", required=True)
    high_priority = api.Integer("Number of low priority (API) requests processed from the server start", required=True)


class WebServerStatusRequestsRate(Api.Schema):
    """ Request processing rate """
    s5 = api.Integer(
        "Average number of requests processed by the server during the last 5 seconds",
        format=api.IntegerFormat.I64, required=True
    )
    s15 = api.Integer(
        "Average number of requests processed by the server during the last 15 seconds",
        format=api.IntegerFormat.I64, required=True
    )
    s30 = api.Integer(
        "Average number of requests processed by the server during the last 30 seconds",
        format=api.IntegerFormat.I64, required=True
    )


class WebServerStatusRequests(Api.Schema):
    """ Information about requests processed by the server """
    count = WebServerStatusRequestsCount(required=True)
    rate = WebServerStatusRequestsRate(required=True)
    processing = api.Integer(
        "Number of requests currently processing by the server", required=True, format=api.IntegerFormat.I64
    )
    queued = api.Integer(
        "Estimated number of requests currently waiting for processing", required=True, format=api.IntegerFormat.I64
    )


class WebServerStatusProcessInfoCPUDetails(Api.Schema):
    """ Detailed information about CPU consumed by the process """
    system = api.Number("Average consumption of system time by the process in percent", required=True)
    user = api.Number("Average consumption of user time by the process in percent", required=True)
    idle = api.Number("Average idle time of the process in percent", required=True)


class WebServerStatusProcessInfoCPU(Api.Schema):
    """ Information about CPU consumed by the process """
    s5 = WebServerStatusProcessInfoCPUDetails(
        "Average consumation of CPU in percent during the last 5 seconds", required=True
    )
    s15 = WebServerStatusProcessInfoCPUDetails(
        "Average consumation of CPU in percent during the last 15 seconds", required=True
    )
    s30 = WebServerStatusProcessInfoCPUDetails(
        "Average consumation of CPU in percent during the last 30 seconds", required=True
    )


class WebServerStatusProcessInfoMemory(Api.Schema):
    """ Information about RAM consumed by the process """
    vms = api.Integer(
        "Amount of available virtual memory for the process, in KiB", required=True, format=api.IntegerFormat.I64
    )
    rss = api.Integer(
        "Amount of the used process memory, in KiB", required=True, format=api.IntegerFormat.I64
    )


class WebServerStatusProcessInfo(Api.Schema):
    """ Information about system resources consumed by the server process """
    cpu = WebServerStatusProcessInfoCPU(required=True)
    memory = WebServerStatusProcessInfoMemory(required=True)


class WebServerStatusProcesses(Api.Schema):
    """ Information about system resources consumed by server processes """
    self = WebServerStatusProcessInfo(required=True)
    service = WebServerStatusProcessInfo(required=True)
    workers = api.Array(WebServerStatusProcessInfo, "Request processing rate", required=True)


class WebServerStatus(Api.Schema):
    """ Information about Web server state """
    host = api.String("Id of Web server processing the request", required=True)
    uptime = api.Number("Server working time in seconds, with millisecond precision", required=True)
    requests = WebServerStatusRequests(required=True)
    processes = WebServerStatusProcesses(required=True)


class TaskStatusStatisticsDraft(Api.Schema):
    """ Number of tasks in status group DRAFT """
    draft = api.Integer("Number of tasks in status DRAFT", required=True, format=api.IntegerFormat.I64)


class TaskStatusStatisticsQueue(Api.Schema):
    """ Number of tasks in status group QUEUE """
    enqueuing = api.Integer("Number of tasks in status ENQUEUING", required=True, format=api.IntegerFormat.I64)
    enqueued = api.Integer("Number of tasks in status ENQUEUED", required=True, format=api.IntegerFormat.I64)


class TaskStatusStatisticsExecute(Api.Schema):
    """ Number of tasks in status group EXECUTE """
    preparing = api.Integer("Number of tasks in status PREPARING", required=True, format=api.IntegerFormat.I64)
    executing = api.Integer("Number of tasks in status EXECUTING", required=True, format=api.IntegerFormat.I64)
    temporary = api.Integer("Number of tasks in status TEMPORARY", required=True, format=api.IntegerFormat.I64)
    finishing = api.Integer("Number of tasks in status FINISHING", required=True, format=api.IntegerFormat.I64)
    suspending = api.Integer("Number of tasks in status SUSPENDING", required=True, format=api.IntegerFormat.I64)
    assigned = api.Integer("Number of tasks in status ASSIGNED", required=True, format=api.IntegerFormat.I64)
    resuming = api.Integer("Number of tasks in status RESUMING", required=True, format=api.IntegerFormat.I64)
    suspended = api.Integer("Number of tasks in status SUSPENDED", required=True, format=api.IntegerFormat.I64)
    stopping = api.Integer("Number of tasks in status STOPPING", required=True, format=api.IntegerFormat.I64)


class TaskStatusStatisticsWait(Api.Schema):
    """ Number of tasks in status group WAIT """
    wait_res = api.Integer("Number of tasks in status WAIT_RES", required=True, format=api.IntegerFormat.I64)
    wait_task = api.Integer("Number of tasks in status WAIT_TASK", required=True, format=api.IntegerFormat.I64)
    wait_time = api.Integer("Number of tasks in status WAIT_TIME", required=True, format=api.IntegerFormat.I64)
    wait_out = api.Integer("Number of tasks in status WAIT_OUT", required=True, format=api.IntegerFormat.I64)
    wait_mutex = api.Integer("Number of tasks in status WAIT_MUTEX", required=True, format=api.IntegerFormat.I64)


class TaskStatusStatisticsFinish(Api.Schema):
    """ Number of tasks in status group FINISH """
    success = api.Integer("Number of tasks in status SUCCESS", required=True, format=api.IntegerFormat.I64)
    releasing = api.Integer("Number of tasks in status RELEASING", required=True, format=api.IntegerFormat.I64)
    released = api.Integer("Number of tasks in status RELEASED", required=True, format=api.IntegerFormat.I64)
    not_released = api.Integer("Number of tasks in status NOT_RELEASED", required=True, format=api.IntegerFormat.I64)
    failure = api.Integer("Number of tasks in status FAILURE", required=True, format=api.IntegerFormat.I64)
    deleted = api.Integer("Number of tasks in status DELETED", required=True, format=api.IntegerFormat.I64)


class TaskStatusStatisticsBreak(Api.Schema):
    """ Number of tasks in status group BREAK """
    exception = api.Integer("Number of tasks in status EXCEPTION", required=True, format=api.IntegerFormat.I64)
    no_res = api.Integer("Number of tasks in status NO_RES", required=True, format=api.IntegerFormat.I64)
    timeout = api.Integer("Number of tasks in status TIMEOUT", required=True, format=api.IntegerFormat.I64)
    stopped = api.Integer("Number of tasks in status STOPPED", required=True, format=api.IntegerFormat.I64)
    expired = api.Integer("Number of tasks in status EXPIRED", required=True, format=api.IntegerFormat.I64)


class TaskStatusStatistics(Api.Schema):
    """ Counters for tasks states """
    draft = TaskStatusStatisticsDraft(required=True)
    queue = TaskStatusStatisticsQueue(required=True)
    execute = TaskStatusStatisticsExecute(required=True)
    wait = TaskStatusStatisticsWait(required=True)
    break_ = TaskStatusStatisticsBreak(required=True)
    finish = TaskStatusStatisticsFinish(required=True)


class ExternalBackupStatistics(Api.Schema):
    """ Information about state of resource backup with ttl=inf in the external storage (MDS) """
    immortal_resource_size = api.Integer("Size in bytes of all resources with ttl=inf", format=api.IntegerFormat.I64)
    external_backup_size = api.Integer("Size in bytes of all backuped resources", format=api.IntegerFormat.I64)


class StorageSizeStatistics(Api.Schema):
    """ Information about storage size """
    storage_ready_resources_size = api.Integer(
        "Size in bytes of all READY-resources on storages", format=api.IntegerFormat.I64
    )
    client_side_ready_resources_size = api.Integer(
        "Size in bytes of all READY-resources on clients", format=api.IntegerFormat.I64
    )
    ready_resources_size = api.Integer(
        "Size in bytes of all READY-resources", format=api.IntegerFormat.I64
    )
    size_pure = api.Integer(format=api.IntegerFormat.I64)
    size_total = api.Integer(format=api.IntegerFormat.I64)
    size_unique = api.Integer(format=api.IntegerFormat.I64)
    disk_total = api.Integer(format=api.IntegerFormat.I64)
    amount_unique = api.Integer(format=api.IntegerFormat.I64)
    disk_free = api.Integer(format=api.IntegerFormat.I64)
    amount_pure = api.Integer(format=api.IntegerFormat.I64)
    delta = api.Integer(format=api.IntegerFormat.I64)
    disk_used = api.Integer(format=api.IntegerFormat.I64)
    amount_total = api.Integer(format=api.IntegerFormat.I64)


class UINotificationCreate(Api.Schema):
    """ Information about new UI-notification """
    severity = api.Enum("Notification importance", required=True, values=ctn.Severity)
    content = api.String("Notification content as HTML in UTF8 encoding", required=True)


class UINotification(UINotificationCreate):
    """ Information about registered UI-notification """
    id = api.String("24-symbolic unique identifier of the notification", required=True)


class QInstance(Api.Schema):
    """ Information about ServiceQ nodes status """
    address = api.String("Node address", required=True)
    port = api.String("Node port", required=True)
    status = api.String("Node status", required=True)


class QueuedTask(Api.Schema):
    """ Information about task in the execution queue """
    id = api.Integer("Global task identifier", required=True, format=api.IntegerFormat.I64)
    url = api.String("Link for retrieving of object's data", required=True)
    priority = task.TaskPriority(required=True)
    clients = api.Array(
        api.Array(api.String),
        "A list of clients suitable to task, sorted by `host_match_score`, in the form of pairs "
        "(client ID, a specialized client's weight (`host_match_score`) for this task)",
        required=True
    )


class QueuedTasksList(common.List):
    """ Task execution queue """
    item = QueuedTask


class TaskTypesNotUsedStatistics(Api.Schema):
    """ List of unused task types """
    type = api.String("Task type name")
    last_run = api.DateTime("Time of the last task running (UTC)")


class HostScore(Api.Schema):
    """ Pair of host and corresponded score """
    host = api.String("Host name")
    score = api.Number("Score")  # FIXME: return to Integer type after SANDBOX-5917


class TaskHosts(Api.Schema):
    """ List of pairs (score, host) for the task """
    task_id = api.Integer("Task id", required=True)
    hosts = api.Array(HostScore, "List of pairs of hosts and its scores", required=True)


class TasksEnqueuedQueueTime(Api.Schema):
    enqueue_time_in_last_hour = api.Number("Tasks enqueued queue time")


class QuotaApi(Api.Schema):
    quota_value = api.Integer("New quota value", required=True)
