from __future__ import absolute_import

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.notification as ctn
from sandbox.common import api

from ..api import Api
from . import common
from . import release as release_schemas


class TaskPriority(Api.Schema):
    """ Task priority """

    # noinspection PyUnresolvedReferences
    class_ = api.Enum(
        "Priority class", required=True, values=list(map(ctt.Priority.Class.val2str, iter(ctt.Priority.Class)))
    )
    # noinspection PyUnresolvedReferences
    subclass = api.Enum(
        "Priority subclass", required=True, values=list(map(ctt.Priority.Subclass.val2str, iter(ctt.Priority.Subclass)))
    )


class TaskTime(Api.Schema):
    """ Base time attributes of task """

    created = api.DateTime("Task creation time, UTC, ISO 8601", required=True)
    updated = api.DateTime("Task update time, UTC, ISO 8601", required=True)


class ClientRef(Api.Schema):
    """ Client info reference """

    id = api.String("Client Id", required=True)
    url = api.String("Url to get full client data", required=True)


class TaskExecutionAction(Api.Schema):
    """ Current task action according to task execution logic """

    message = api.String("Description of action")
    duration = api.Integer("Time in seconds from the action start")


class TaskExecution(Api.Schema):
    """ Task execution info """

    client = ClientRef()
    current = api.Number("Time in seconds from task execution start")
    estimated = api.Number("Estimated execution time in seconds")
    action = TaskExecutionAction()
    ps_url = api.String("Url to get list of task processes")
    actions_url = api.String("URL to list of custom actions this task is currently performing")
    shell_url = api.String("Url to Shell to debug task in UI")
    started = api.DateTime("Task start time, UTC, ISO 8601")
    finished = api.DateTime("Task finish time, UTC, ISO 8601")


class TaskInterval(Api.Schema):
    """ Task interval (one stage of task execution) """
    start = api.DateTime("Start time of interval")
    duration = api.Integer("Duration of interval in seconds")
    consumption = api.Integer("Quota consumption during interval, milliQP")
    pool = api.String("Quota pool name")


class TaskIntervals(Api.Schema):
    """ Task intervals """
    execute = api.Array(TaskInterval, "Execution intervals")
    queue = api.Array(TaskInterval, "Queue intervals")
    wait = api.Array(TaskInterval, "Wait intervals")


class TaskSubCollection(Api.Schema):
    """ Description of collection of objects owned by task """

    url = api.String("Url to get collection", required=True)
    count = api.Integer("Collection size", required=True)


class TaskRequirementsRamdisk(Api.Schema):
    """ Required RAM drive info """

    type = api.Enum("RAM drive type", values=ctm.RamDriveType, required=True)
    size = api.Integer("RAM drive size in bytes", format=api.IntegerFormat.I64, required=True)


class TaskSemaphoreAcquire(Api.Schema):
    """ Define semaphore to acquire """

    name = api.String("Semaphore's name", required=True)
    weight = api.Integer("Task's weight in the semaphore")
    capacity = api.Integer("Semaphore's capacity")


class TaskSemaphores(Api.Schema):
    """ Required semaphores """

    acquires = api.Array(TaskSemaphoreAcquire, "Semaphores to acquire", required=True)
    release = api.Array(api.String, "List of statuses/groups of statuses in which semaphore will be released")


class TaskSemaphoreAcquireUpdate(TaskSemaphoreAcquire):
    """ Define semaphore to acquire """

    public = api.Boolean("Public semaphore")


class TaskSemaphoresUpdate(TaskSemaphores):
    """ Required semaphores """

    acquires = api.Array(TaskSemaphoreAcquireUpdate, "Semaphores to acquire", required=True)


class BucketReserve(Api.Schema):
    """ Space to reserve in MDS bucket for resources the task going to create """

    bucket = api.String("Bucket")
    size = api.Integer("Space to reserve in bytes")


class TaskRequirementsBase(Api.Schema):
    """ Base task requirements """
    platform = api.String("Required platform")
    cpu_model = api.String("Required CPU model")
    host = api.String("Client Id")
    disk_space = api.Integer("Required disk space in bytes", format=api.IntegerFormat.I64)
    ram = api.Integer("Required RAM in bytes", format=api.IntegerFormat.I64)
    ramdrive = TaskRequirementsRamdisk()
    client_tags = api.String("Client tag expression")
    semaphores = TaskSemaphores("Semaphores")
    privileged = api.Boolean("Privileged")
    dns = api.Enum("Type of resolving config", values=ctm.DnsType)
    cores = api.Integer("CPU cores")
    tasks_resource = api.Id("Task code resource id of type SANDBOX_TASKS_(ARCHIVE|BINARY|IMAGE)", required=False)
    caches = api.Object("Caches")
    container_resource = api.Id("Container resource id of type LXC_CONTAINER or inherited", required=False)
    porto_layers = api.Array(api.Integer, "Porto layers")
    resources_space_reserve = api.Array(BucketReserve, "Resources space reserve")


class RequiredResources(TaskSubCollection):
    """ Required resources """

    ids = api.Array(api.Id, "Resources IDs")


class TaskRequirements(TaskRequirementsBase):
    """ Tasks requirements """

    resources = RequiredResources("Required resources")


class BaseNotifications(Api.Schema):
    """ Base notification settings for task and schedulers """

    transport = api.Enum("Notification transport", values=ctn.Transport, required=True)
    recipients = api.Array(api.String, "List of logins notifications should be sent to", required=True)
    check_status = api.Enum("Juggler check status", values=ctn.JugglerStatus)
    juggler_tags = api.Array(api.String, "List of tags for Juggler")


class TaskNotifications(BaseNotifications):
    """ Notification settings """

    statuses = api.Array(api.Enum(values=ctt.Status), "Task statuses when notifications should be sent", required=True)


class DiskUsageInfo(Api.Schema):
    """ Disk usage during task execution info """

    max = api.Integer("Max disk usage in bytes during task execution in bytes", format=api.IntegerFormat.I64, minimum=0)
    last = api.Integer("Disk usage in bytes after task execution", format=api.IntegerFormat.I64, minimum=0)


class TaskResults(Api.Schema):
    """ Task results """

    info = api.String("Info from task execution")
    traceback = api.String("Python traceback")
    disk_usage = DiskUsageInfo(required=True)


class TaskRelease(Api.Schema):
    """ Task release info """

    id = api.Id(required=False)
    resources = api.Array(release_schemas.ReleaseResource, "Resources to release")
    template = release_schemas.ReleaseTemplate("Release template")
    url = api.String("Url to get release data")
    type = api.Enum("Release type", values=ctt.ReleaseStatus)


class TaskReportListItem(Api.Schema):
    label = api.String("Report name")
    title = api.String("Report tab title")
    url = api.String("URL to obtain report from")


class Uniqueness(Api.Schema):
    """ Deduplication uniqueness """

    key = api.String("Deduplication unique key")
    excluded_statuses = api.Array(
        api.String, "Task statuses, excluded from uniqueness detection", default=[ctt.Status.DELETED]
    )


class TaskBase(Api.Schema):
    """ Task's base fields """

    id = api.Id("Task Id")
    url = api.String("Link to the task object")
    parent = common.ObjectRef("Parent task")
    rights = api.Enum("Maximum available right of the current user on this task", values=ctu.Rights)
    priority = TaskPriority()
    important = api.Boolean("Task is important")
    status = api.String("Task status")
    type = api.String("Task type")
    scores = api.Integer("Success score of last 10 task runs", minimum=1, maximum=10)
    description = api.String("Task description")
    author = api.String("Task author")
    owner = api.String("Task owner")
    time = TaskTime("Time attributes of task")
    execution = TaskExecution("Task execution info")
    intervals = TaskIntervals("Task intervals (duration of time spent in QUEUE, WAIT, EXECUTE stages)")
    requirements = TaskRequirements()
    kill_timeout = api.Integer("Max time in seconds that task is allowed to execute")
    fail_on_any_error = api.Boolean("Switch task to FAILURE in case of any error")
    tasks_archive_resource = api.Id("Task code resource Id of type SANDBOX_TASKS_(ARCHIVE|BINARY)", required=False)
    dump_disk_usage = api.Boolean("Detailed disk usage statistics")
    tcpdump_args = api.String("Tcpdump arguments that are used for network packets logging")
    tags = api.Array(api.String, "Task tags")
    hidden = api.Boolean("Task is hidden flag")
    max_restarts = api.Integer("Max restarts after TEMPORARY")
    expires = api.Integer("Task expires in N seconds from start")
    unique_key = api.String("Deduplication unique key")
    notifications = api.Array(TaskNotifications, "Notification settings")
    input_parameters = api.Object("Input parameters")
    output_parameters = api.Object("Output parameters (SDK2 only)")
    suspend_on_status = api.Array(api.Enum(values=ctt.Status.Group.SUSPEND_ON_STATUS), "Statuses for suspending")
    score = api.Integer("Task priority score")
    push_tasks_resource = api.Boolean("Push tasks resource to subtasks")
    results = TaskResults("Task results")
    release = TaskRelease()
    reports = api.Array(TaskReportListItem, "User-defined reports for a task (footer, ...)")
    se_tag = api.String("Limiting tag (deprecated, use semaphores instead)")
    se_tags = api.Array(api.String, "Limiting tags (deprecated, use semaphores instead)")
    uniqueness = Uniqueness("Deduplication uniqueness")
    hints = api.Array(api.String, "Task hint(s)")
    enable_yav = api.Boolean("Task use yav")
    scheduler = common.ObjectRef("Scheduler that created the task")
    template_alias = api.String("Template alias that created the task")
    children = api.Array(api.Object, "Dictionary of children:'task Id' => 'task state'")


class WorkDiskUsageInfo(DiskUsageInfo):
    """ Disk usage during task execution (only work directory and unregistered resources) """

    resources = api.Integer(
        "Size of unregistered resources in bytes downloaded by task execution",
        format=api.IntegerFormat.I64, minimum=0
    )


class TaskExecutionUpdate(Api.Schema):
    """ Task execution update data """
    description = api.String("Description of current execution step", default=ctm.NotExists)
    disk_usage = DiskUsageInfo()
    work_disk_usage = WorkDiskUsageInfo()


class TaskRequirementsUpdate(TaskRequirementsBase):
    """ Tasks requirements update info """

    semaphores = TaskSemaphoresUpdate("Semaphores")
    disk_space = TaskRequirementsBase.disk_space(required=False)


class TaskFieldValidateItem(Api.Schema):
    """ Field validation data """

    name = api.String("Field name", required=True)
    value = api.Object("Field data", required=True)


class TaskOutputFieldValidateItem(TaskFieldValidateItem):
    """ Output field validation data """

    reset_on_restart = api.Boolean("Reset on restart")


class TaskLogItem(Api.Schema):
    """ Task log info """

    name = api.String("File or directory name", required=True)
    node_type = api.Enum("Type of url destination", values=ctm.FileType, required=True)
    url = api.String("Url to get data", required=True)
    size = api.Integer("Log size in bytes", required=True, format=api.IntegerFormat.I64)


class Task(TaskBase):
    """ Detailed task info """

    sdk_version = api.Integer("Task's SDK version")
    lock_host = api.String("Host executing task")
    short_task_result = api.Object("Short task result. Filled for requests from web")
    resource = TaskSubCollection("Created resources")
    dependant = TaskSubCollection("Dependent tasks")
    audit = TaskSubCollection("Audit records for task")
    queue = TaskSubCollection("Queue state for task")
    logs = api.Array(TaskLogItem, "Task logs info")
    effective_client_tags = api.String("Effective client tags for last enqueueing")
    pools = api.Array(api.String, "Task enqueue affected pools")


class TaskListItem(TaskBase):
    """ Task item """

    requirements = TaskRequirements()
    short_task_result = api.Object("Short task result. Filled for requests from web")
    context = api.Object("Task context")


class TaskUpdate(Api.Schema):
    """ Task update info """

    priority = TaskPriority()
    important = api.Boolean("Task is important")
    description = api.String("Task description")
    owner = api.String("Task owner")
    kill_timeout = api.Integer("Max time in seconds that task is allowed to execute")
    fail_on_any_error = api.Boolean("Switch task to FAILURE in case of any error")
    tasks_archive_resource = api.Id(
        "Task code resource ID (deprecated, use requirements.tasks_resource instead)", required=False,
    )
    hidden = api.Boolean("Task is hidden flag (deprecated)")
    se_tag = api.String("Limiting tag (deprecated, use semaphores instead)")
    tags = api.Array(api.String, "Custom task tags")
    hints = api.Array(api.String, "Task hints")
    max_restarts = api.Integer("Max restarts after TEMPORARY")
    requirements = TaskRequirementsUpdate
    notifications = api.Array(TaskNotifications, "Notification settings")
    custom_fields = api.Array(TaskFieldValidateItem)
    suspend_on_status = api.Array(api.Enum(values=ctt.Status.Group.SUSPEND_ON_STATUS), "Statuses for suspending")
    push_tasks_resource = api.Boolean("Push tasks resource to subtasks")
    score = api.Integer("Task priority score")


class TaskNew(TaskUpdate):
    """ Data for new task initialization """

    author = api.String("The task will be run on behalf of this user")
    type = api.String("Task type. Conflicts with 'source' and 'scheduler_id'")
    source = api.Integer("Task Id to copy. Conflicts with 'type' and 'scheduler_id'", format=api.IntegerFormat.I64)
    scheduler_id = api.Integer("Scheduler Id that provides task template. Conflicts with 'type' and 'source'")
    template_alias = api.String("template alias that provides task template. Conflicts with 'type' and 'source'")
    regular_schedule = api.Boolean("Task is scheduled by service process")
    children = api.Boolean("Mark created task as child for current session's task")
    context = api.Object("Initial context")
    uniqueness = Uniqueness("Deduplication uniqueness")


class TaskCustomFieldMetaBase(Api.Schema):
    """ Custom field info base class """

    type = api.String("Field type that defines representation of field value")
    name = api.String("Field name (Id)", required=True)
    title = api.String("Field title, short description")
    description = api.String("Field description")
    required = api.Boolean("Is the field required")
    sub_fields = api.Object("Dictionary <field value> => <array of active subfields>")
    modifiers = api.Object("Representation/validation modifiers")
    context = api.Object("Representation context info")
    output = api.Boolean("Is an output parameter")


class TaskCustomFieldMeta(TaskCustomFieldMetaBase):
    """ Custom field info with an actual value of the field """

    value = api.Object("Field value", required=True)


class TaskReport(Api.Schema):
    """ Custom footer info """

    content = api.Object("User data to display", required=True)
    helper = api.String("JavaScript view helper name", required=False)


class TaskAuditItem(Api.Schema):
    """ Task audit item """

    time = api.DateTime("Event time, UTC, ISO 8601", required=True)
    message = api.String("Event description")
    status = api.Enum("New task state", values=ctt.Status)
    author = api.String("Event initiator")
    task_id = api.Id("Task Id")
    target = api.String("Host that handled event", required=True)
    request_id = api.String("Request Id")
    source = api.String("Network address of event initiator")
    iface = api.Enum("Interface of event creation", values=ctt.RequestSource, required=True)
    wait_targets = api.Object("Dictionary of time, tasks or resources awaited by the task")


class TaskHardware(Api.Schema):
    """ Options for requests to solomon to get hardware statistics """

    project = api.String("Solomon project name", required=True)
    cluster = api.String("Solomon cluster name", required=True)
    service = api.String("Solomon service name", required=True)
    host = api.String("FQDN or IP-address of host that performed the task", required=True)
    b = api.DateTime("Execution start time, UTC, ISO 8601", required=True)
    e = api.DateTime("Execution finish time, UTC, ISO 8601")
    info = api.String("Sensor info")
    partition = api.String("Sensor partition")
    sensor = api.String("Sensor name")
    path = api.String("Same as sensor name")
    servant = api.String("Samogon servant")


class TaskAuditHostsItem(Api.Schema):
    """ Task executing host info """

    host = api.String("FQDN or IP-address of host that performed the task", required=True)
    from_ = api.DateTime("Execution start time, UTC, ISO 8601", required=True)
    till = api.DateTime("Execution finish time, UTC, ISO 8601")
    partition = api.String("Drive partition with task artifacts", required=True)
    cpu_usage = TaskHardware("Cpu usage", required=True)
    virt_mem = TaskHardware("Virtual memory usage", required=True)
    disk_usage = TaskHardware("Disk usage", required=True)
    client_id = api.String("Client id", required=True)


class TaskAuditNew(Api.Schema):
    """ Task audit info """

    message = api.String("Event description")
    status = api.Enum("New task state", values=ctt.Status)
    force = api.Boolean("Do not validate status transition")
    expected_status = api.Enum("Current task status, use to prevent conflict", values=ctt.Status)
    lock = api.String("Host lock")
    wait_targets = api.Object("Dictionary of time, tasks or resources awaited by the task")


class QueuePosition(Api.Schema):
    """ Task in queue position info """
    index = api.Integer("Current position", required=True)
    size = api.Integer("Queue size", required=True)


class TaskQueueItem(Api.Schema):
    """ Client info that is suitable for the task """

    client = ClientRef(required=True)
    alive = api.Boolean("Client is alive flag", required=True)
    platform = api.Boolean("Platform is suitable flag", required=True)
    cpu = api.Boolean("CPU is suitable flag", required=True)
    free_space = api.Boolean("It's enough free space on the host", required=True)
    position = QueuePosition


class TaskFieldValidateResultItem(Api.Schema):
    """ Field validation result """

    name = api.String("Field name", required=True)
    status = api.Enum("Validation status", values=ctt.FieldValidationStatus, required=True)
    message = api.String("Error description")


class TaskTimeTriggerPeriod(Api.Schema):
    """ Time trigger period info """

    original = api.Integer("Time in seconds, specified when creating", required=True)
    remaining = api.Integer("Remaining time in seconds", required=True)


class TaskTimeTriggerTime(Api.Schema):
    """ Time trigger time info """

    registered = api.DateTime("Registration time, UTC, ISO 8601", required=True)
    expected = api.DateTime("Expected time, UTC, ISO 8601", required=True)


class TaskTimeTrigger(Api.Schema):
    """ Time trigger info """

    period = TaskTimeTriggerPeriod(required=True)
    time = TaskTimeTriggerTime(required=True)


class TaskTimeTriggerCreate(Api.Schema):
    """ Parameters of time trigger to create """

    period = api.Integer("Period in seconds", required=True)


class TaskTrigger(Api.Schema):
    """ Base trigger parameters """

    targets = api.Array(api.Integer, " List of task identifiers", required=True)
    wait_all = api.Boolean("Wait for all tasks or any", required=True)


class TaskTaskTrigger(TaskTrigger):
    """ Parameters of task status trigger """

    statuses = api.Array(api.Enum(values=ctt.Status), "List of expected task statuses", required=True)


class TaskTaskTriggerCreate(TaskTaskTrigger):
    """ Parameters of task status trigger to create """

    timeout = api.Integer("Time for waiting tasks")


class TaskOutputTriggerCreate(Api.Schema):
    """ Parameters of output trigger to create """

    targets = api.Map(
        api.Array(api.String, "Output fields for waiting"), "Dict of task id : list of output fields", required=True
    )
    timeout = api.Integer("Time for waiting tasks")
    wait_all = api.Boolean("Wait for all tasks or any", required=True)


class TaskList(common.List):
    item = TaskListItem


class CurrentContextValue(Api.Schema):
    """ Schema for updating context field """

    key = api.String("Name of field to update", required=True)
    value = api.DataType("New value of field")


class TaskReleaseCreate(Api.Schema):
    """ Release information """

    author = api.String("Release author", required=True)
    release_status = api.String("Release status", required=True)
    message_body = api.String("Release message")
    changelog = api.String("Changelog")
    creation_time = api.DateTime("Creation time")
