from __future__ import absolute_import

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.template as cttmpl
from sandbox.common import api

from . import task
from ..api import Api

from . import common


class TemplateBase(Api.Schema):
    description = api.String("Template description")
    shared_with = api.Array(api.String, "List of users or groups, who can edit template")


class TemplateCreate(TemplateBase):
    alias = api.String("Template alias")
    task_id = api.Id("Create template from task", required=False)
    template_alias = api.String("Create template from another template", required=False)
    task_type = api.String("The task name for the template")
    tasks_resource = api.Id("Tasks resource ID", required=False)
    tasks_resource_filter = api.Object("Filter for tasks resource")


class TemplateFieldsMetaBase(Api.Schema):
    hide = api.Boolean("Field must be hidden in UI")
    default_from_code = api.Boolean("Use default value from code")
    filter = api.Object("Filter for resources or tasks")


class TemplateFieldsMeta(task.TaskCustomFieldMetaBase, TemplateFieldsMetaBase):
    default = api.DataType("Default value from code")
    order = api.Integer("Order of parameter")


class TaskCustomParameter(Api.InlineSchema):
    meta = TemplateFieldsMeta("Parameter meta information")
    value = api.DataType("Parameter value")


class TaskCustomParameterUpdate(Api.InlineSchema):
    meta = TemplateFieldsMetaBase("Parameter meta information")
    value = api.Object("Parameter value")


class TemplateTaskRequirements(Api.Schema):
    host = TaskCustomParameter("Client Id", overrides={"value": api.String("Host value")})
    disk_space = TaskCustomParameter(
        "Required disk space in bytes", overrides={"value": api.Integer("Disk space value")}
    )
    ram = TaskCustomParameter("Required RAM in bytes", overrides={"value": api.Integer("Ram value")})
    ramdrive = TaskCustomParameter(
        "Ramdisk requirements", overrides={"value": task.TaskRequirementsRamdisk("Ramdisk value")}
    )
    client_tags = TaskCustomParameter("Client tag expression", overrides={"value": api.String("Client tags")})
    semaphores = TaskCustomParameter("Semaphores", overrides={"value": task.TaskSemaphores("Semaphores")})
    privileged = TaskCustomParameter("Privileged", overrides={"value": api.Boolean("Privileged value")})
    dns = TaskCustomParameter(
        "Type of resolving config", overrides={"value": api.Enum("Dns value", values=ctm.DnsType)}
    )
    cores = TaskCustomParameter("CPU cores", overrides={"value": api.Integer("Cores value")})
    tasks_resource = TaskCustomParameter(
        "Tasks resource", overrides={"value": api.Id("Tasks resource ID", required=False)}
    )
    container_resource = TaskCustomParameter(
        "Tasks container", overrides={"value": api.Id("Tasks container ID", required=False)}
    )
    porto_layers = TaskCustomParameter(
        "List of porto layers", overrides={"value": api.Array(api.Id, "Porto layers values")}
    )
    resources_space_reserve = TaskCustomParameter(
        "Resources space reserve", overrides={"value": api.Array(task.BucketReserve, "Resources space reserve")}
    )


class TemplateTaskCommonParameters(Api.Schema):
    owner = TaskCustomParameter("Task owner", overrides={"value": api.String("Owner value")})
    description = TaskCustomParameter("Task description", overrides={"value": api.String("Description value")})
    priority = TaskCustomParameter("Task priority", overrides={"value": task.TaskPriority("Priority value")})
    kill_timeout = TaskCustomParameter(
        "Max time in seconds that task is allowed to execute", overrides={"value": api.Integer("Kill timeout value")}
    )
    fail_on_any_error = TaskCustomParameter(
        "Switch task to FAILURE in case of any error", overrides={"value": api.Boolean("Fail on any error value")}
    )
    hidden = TaskCustomParameter("Task is hidden flag", overrides={"value": api.Boolean("Hidden value")})
    tags = TaskCustomParameter("Custom task tag(s)", overrides={"value": api.Array(api.String("Tags value"))})
    max_restarts = TaskCustomParameter(
        "Max restarts after TEMPORARY", overrides={"value": api.Integer("Max restarts value")}
    )
    suspend_on_status = TaskCustomParameter(
        "Statuses for suspending",
        overrides={"value": api.Array(api.Enum(values=ctt.Status.Group.SUSPEND_ON_STATUS), "Suspend statuses")}
    )
    score = TaskCustomParameter(
        "Task priority score", overrides={"value": api.Integer("Task priority score")}
    )
    push_tasks_resource = TaskCustomParameter(
        "Push tasks resource to subtasks", overrides={"value": api.Boolean("Push tasks resource to subtasks")}
    )
    expires = TaskCustomParameter(
        "Task expires in N seconds from start", overrides={"value": api.Integer("Expires value")}
    )
    notifications = TaskCustomParameter(
        "Notification settings", overrides={"value": api.Array(task.TaskNotifications, "Notifications")}
    )
    dump_disk_usage = TaskCustomParameter(
        "Dump disk usage", overrides={"value": api.Boolean("Dump disk usage")}
    )
    tcpdump_args = TaskCustomParameter(
        "TCP dump args", overrides={"value": api.String("TCP dump args value")}
    )


class TemplateTaskRequirementsUpdate(Api.Schema):
    host = TaskCustomParameterUpdate("Client Id", overrides={"value": api.String("Host value")})
    disk_space = TaskCustomParameterUpdate(
        "Required disk space in bytes", overrides={"value": api.Integer("Disk space value")}
    )
    ram = TaskCustomParameterUpdate("Required RAM in bytes", overrides={"value": api.Integer("Ram value")})
    ramdrive = TaskCustomParameterUpdate(
        "Ramdisk requirements", overrides={"value": task.TaskRequirementsRamdisk("Ramdisk value")}
    )
    client_tags = TaskCustomParameterUpdate("Client tag expression", overrides={"value": api.String("Client tags")})
    semaphores = TaskCustomParameterUpdate("Semaphores", overrides={"value": task.TaskSemaphoresUpdate("Semaphores")})
    privileged = TaskCustomParameterUpdate("Privileged", overrides={"value": api.Boolean("Privileged value")})
    dns = TaskCustomParameterUpdate(
        "Type of resolving config", overrides={"value": api.Enum("Dns value", values=ctm.DnsType)}
    )
    cores = TaskCustomParameterUpdate("CPU cores", overrides={"value": api.Integer("Cores value")})
    tasks_resource = TaskCustomParameterUpdate(
        "Tasks resource", overrides={"value": api.Id("Tasks resource ID", required=False)}
    )
    container_resource = TaskCustomParameterUpdate(
        "Tasks container", overrides={"value": api.Id("Tasks container ID", required=False)}
    )
    porto_layers = TaskCustomParameterUpdate(
        "List of porto layers", overrides={"value": api.Array(api.Id, "Porto layes values")}
    )


class TemplateTaskCommonParametersUpdate(Api.Schema):
    owner = TaskCustomParameterUpdate("Task owner", overrides={"value": api.String("Owner value")})
    description = TaskCustomParameterUpdate("Task description", overrides={"value": api.String("Description value")})
    priority = TaskCustomParameterUpdate("Task priority", overrides={"value": task.TaskPriority("Priority value")})
    kill_timeout = TaskCustomParameterUpdate(
        "Max time in seconds that task is allowed to execute", overrides={"value": api.Integer("Kill timeout value")}
    )
    fail_on_any_error = TaskCustomParameterUpdate(
        "Switch task to FAILURE in case of any error", overrides={"value": api.Boolean("Fail on any error value")}
    )
    hidden = TaskCustomParameterUpdate("Task is hidden flag", overrides={"value": api.Boolean("Hidden value")})
    tags = TaskCustomParameterUpdate("Custom task tag(s)", overrides={"value": api.Array(api.String("Tags value"))})
    max_restarts = TaskCustomParameterUpdate(
        "Max restarts after TEMPORARY", overrides={"value": api.Integer("Max restarts value")}
    )
    suspend_on_status = TaskCustomParameterUpdate(
        "Statuses for suspending",
        overrides={"value": api.Array(api.Enum(values=ctt.Status.Group.SUSPEND_ON_STATUS), "Suspend statuses")}
    )
    score = TaskCustomParameterUpdate(
        "Task priority score", overrides={"value": api.Integer("Task priority score")}
    )
    push_tasks_resource = TaskCustomParameter(
        "Push tasks resource to substasks", overrides={"value": api.Boolean("Push tasks resource to substasks")}
    )
    expires = TaskCustomParameterUpdate(
        "Task expires in N seconds from start", overrides={"value": api.Integer("Expires value")}
    )
    notifications = TaskCustomParameterUpdate(
        "Notification settings", overrides={"value": api.Array(task.TaskNotifications, "Notifications")}
    )


class TaskFields(Api.Schema):
    input = api.Map(TaskCustomParameter, "Task input parameters")
    common = TemplateTaskCommonParameters("Task common parameters")


class TemplateTask(Api.Schema):
    type = api.String("The task name for the template")
    parameters = TaskFields("Task parameters")
    requirements = TemplateTaskRequirements("Task requirements")


class TaskFieldsUpdate(Api.Schema):
    input = api.Map(TaskCustomParameterUpdate, "Task input parameters")
    common = TemplateTaskCommonParametersUpdate("Task common parameters")


class TemplateTaskUpdate(Api.Schema):
    parameters = TaskFieldsUpdate("Task parameters")
    requirements = TemplateTaskRequirementsUpdate("Task requirements")


class TemplateUpdate(TemplateBase):
    task = TemplateTaskUpdate("Task")


class Time(Api.Schema):
    created = api.DateTime("Time of template creation", required=True)
    updated = api.DateTime("Time of template last update", required=True)


class Template(TemplateBase):
    alias = api.String("Template alias")
    task = TemplateTask("Task")
    author = api.String("Template author")
    status = api.Enum("Template status", values=cttmpl.Status)
    time = Time("Time template properties")
    favorite = api.Boolean("Template added to favorites")


class TemplateList(common.List):
    item = Template


class TaskParameterAudit(Api.Schema):
    old = TaskCustomParameter("Old parameter information and value")
    new = TaskCustomParameter("New parameter information and value")


class TemplatePropertyAudit(Api.Schema):
    old = api.DataType("Old parameter information and value")
    new = api.DataType("New parameter information and value")


class ParametersAudit(Api.Schema):
    input = api.Map(TaskParameterAudit, "Input parameters audit")
    common = api.Map(TaskParameterAudit, "Common parameters audit")


class TaskAudit(Api.Schema):
    parameters = ParametersAudit("Parameters audit")
    requirements = api.Map(TaskParameterAudit, "Requirements audit")


class TemplateAudit(Api.Schema):
    template_alias = api.String("Template alias")
    author = api.String("Update author", required=True)
    date = api.DateTime("Update time", required=True)
    task = TaskAudit("Task audit")
    properties = api.Map(TemplatePropertyAudit, "Template properties audit")


class TemplateAuditList(common.List):
    item = TemplateAudit
