"""
Operations on semaphores
"""
from __future__ import absolute_import

from sandbox.common import api

from .api import Api
from .common import FilterBase
from . import schemas


class SemaphoreList(Api.Path("/semaphore")):
    class Get(FilterBase):
        """ List of semaphores """
        name = api.String("Filter by element's name substring")
        owner = api.String("Search by owner name")
        shared = api.String("Search by name of group, the element is shared with")
        auto = api.Boolean("Search by automatically creation")
        group = api.String("Search by semaphore group")

        class Ok(api.Response):
            """ Search result """
            schema = schemas.semaphore.SemaphoreList
            x_matched_records = api.Header(api.Integer("Total amount of records matched by a query"))

    class Post(api.Post):
        """ Create a new semaphore """
        data = schemas.semaphore.SemaphoreCreate(scope=api.Scope.BODY)

        class Created(api.Response):
            """ Object has been created successfully """
            schema = schemas.semaphore.Semaphore
            location = api.Header("URL to fetch just created object's data")


class Semaphore(Api.Path("/semaphore/{id}")):
    class Get(api.Get):
        """ Semaphore item object fetch """
        id = api.Id(scope=api.Scope.PATH)

        class Ok(api.Response):
            """ Item object """
            schema = schemas.semaphore.Semaphore

        class NotFound(api.Response):
            """ No object with ID provided can be found """

    class Put(api.Put):
        """ Semaphore item object update """
        id = api.Id(scope=api.Scope.PATH)
        data = schemas.semaphore.SemaphoreUpdate(scope=api.Scope.BODY)

        class Ok(api.Response):
            """
            In case of request header `X-Request-Updated-Data` was passed, the updated item's object will be returned
            """
            schema = schemas.semaphore.Semaphore

        class NoContent(api.Response):
            """ Object has been updated successfully """

        class NotFound(api.Response):
            """ No object with ID provided can be found """

    class Delete(api.Delete):
        """ Delete the given item """
        id = api.Id(scope=api.Scope.PATH)

        class NoContent(api.Response):
            """ An object has been deleted successfully """

        class NotFound(api.Response):
            """ No object with ID provided can be found """

        class Locked(api.Response):
            """ Semaphore is busy """


class SemaphoreAudit(Api.Path("/semaphore/{id}/audit")):
    class Get(api.Get):
        """ Semaphore audit """
        id = api.Id("Semaphore Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.semaphore.SemaphoreAuditItem)
