""" Information about service status, statistics and administrative operations """

from __future__ import absolute_import

import sandbox.common.types.misc as ctm
import sandbox.common.types.user as ctu
from sandbox.common import api
from sandbox.common import rest

from .api import Api
from . import schemas
from .common import FilterBase


class ServiceStatusQInstances(Api.Path("/service/status/q/instances")):
    class Get(api.Get):
        """ Information about ServiceQ nodes status """
        class Ok(api.Response):
            schema = api.Array(schemas.service.QInstance)


class ServiceStatusThreads(Api.Path("/service/status/threads")):
    class Get(api.Get):
        """ Information about status of service threads """
        class Ok(api.Response):
            schema = api.Array(schemas.service.ServiceThreadStatus)


class ServiceStatusOperationMode(Api.Path("/service/status/operation_mode")):
    class Get(api.Get):
        """ Current functional mode of the service """
        class Ok(api.Response):
            schema = api.Enum(values=ctm.OperationMode)

    class Put(api.Put):
        """ Change current functional mode of the service """
        __security__ = ctu.Restriction.ADMIN

        mode = api.Enum("Functional mode", values=ctm.OperationMode, required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            pass


class ServiceStatusDatabaseShards(Api.Path("/service/status/database/shards")):
    class Get(api.Get):
        """ Information about state of the MongoDB processes """
        class Ok(api.Response):
            schema = api.Array(schemas.service.DatabaseShard)


class ServiceStatusDatabaseSize(Api.Path("/service/status/database/size")):
    class Get(api.Get):
        """ Information about DB size """
        class Ok(api.Response):
            schema = schemas.service.DatabaseSize


class ServiceStatusDatabaseCurrentOp(Api.Path("/service/status/database/current_op")):
    class Get(api.Get):
        """ Information about total duration of the current DB operations """
        class Ok(api.Response):
            schema = api.Array(schemas.service.DatabaseCurrentOp)


class ServiceStatusServer(Api.Path("/service/status/server")):
    class Get(api.Get):
        """ Information about state of Web-server processing request """
        class Ok(api.Response):
            schema = schemas.service.WebServerStatus


class ServiceStatisticsTaskStatus(Api.Path("/service/statistics/task/status/{stat_type}")):
    class Get(api.Get):
        """ Absolute counters for tasks by statuses """
        stat_type = api.Enum("Statistics type", values=["absolute", "delta"], required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.service.TaskStatusStatistics


class ServiceStatisticsTaskTypesNotUsed(Api.Path("/service/statistics/task/types/not_used")):
    class Get(api.Get):
        """ Task type list that have not running more than `days_ago` days """
        days_ago = api.Integer("Number of days", scope=api.Scope.QUERY, default=180, minimum=-1)

        class Ok(api.Response):
            schema = api.Array(schemas.service.TaskTypesNotUsedStatistics)


class ServiceStatisticsStorageExternalBackup(Api.Path("/service/statistics/storage/external_backup")):
    class Get(api.Get):
        """ Information about state of resource backup with ttl=inf in the external storage (MDS) """
        class Ok(api.Response):
            schema = schemas.service.ExternalBackupStatistics


class ServiceStatisticsStorageSize(Api.Path("/service/statistics/storage/size")):
    class Get(api.Get):
        """ Information about storage size """
        class Ok(api.Response):
            schema = schemas.service.StorageSizeStatistics


class ServiceUINotificationList(Api.Path("/service/ui/notification")):
    class Get(api.Get):
        """ List of registered UI notification """
        class Ok(api.Response):
            schema = api.Array(schemas.service.UINotification)

    class Post(api.Post):
        """ Register new UI notification """
        __security__ = ctu.Restriction.ADMIN

        notification = schemas.service.UINotificationCreate("UI notification", required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            schema = schemas.service.UINotification
            location = api.Header(api.String("Link to the created notification"))


class ServiceUINotification(Api.Path("/service/ui/notification/{id}")):
    class Put(api.Put):
        """ Update UI notification """
        __security__ = ctu.Restriction.ADMIN

        id = api.String("24-symbolic unique identifier of the notification", required=True, scope=api.Scope.PATH)
        body = schemas.service.UINotificationCreate("UI notification", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            pass

    class Delete(api.Delete):
        """ Remove UI notification """
        __security__ = ctu.Restriction.ADMIN

        id = api.String("24-symbolic unique identifier of the notification", required=True, scope=api.Scope.PATH)

        class NoContent(api.Response):
            """ Deleted successfully """


class ServiceQ(Api.Path("/service/q")):
    class Get(FilterBase):
        """ Task execution queue """
        client = api.String("Client identifier")

        class Ok(api.Response):
            schema = schemas.service.QueuedTasksList


class ServiceShortifyClient(Api.Path("/service/shortify/client")):
    class Post(api.Post):
        """ Service for getting short list from list of client identifiers """
        clients_list = api.Array(api.String, "Client identifier list to shortify", scope=api.Scope.BODY)

        class Created(api.Response):
            __content__ = rest.Client.PLAINTEXT
            schema = api.String("Shortified client list")


class ServiceTimeCurrent(Api.Path("/service/time/current")):
    class Get(api.Get):
        """ Current server time """

        class Ok(api.Response):
            __content__ = rest.Client.PLAINTEXT
            schema = api.String("Current server time (UTC)")


class ServiceResources(Api.Path("/service/resources")):
    class Get(api.Get):
        """ Commonly used environment resources cache """
        type = api.String("Resource type")
        platform = api.String("Resource platform", default="")
        version = api.String("Resource version", default="")

        class Ok(api.Response):
            schema = api.Integer("Matching resource id")


class ServiceUnavailable(Api.Path("/service/unavailable")):
    class Get(api.Get):
        class ServiceUnavailable(api.Response):
            """ Always respond 503 SERVICE_UNAVAILABLE """
            pass


class ServiceTVM(Api.Path("/service/tvm")):
    class Get(api.Get):
        class ServiceUnavailable(api.Response):
            """ Check that TVM tickets are fetched properly """
            pass


class ServiceProxyBannedNetworks(Api.Path("/service/banned_networks")):
    class Get(api.Get):
        """ List of networks banned in proxy """
        __security__ = ctu.Restriction.ADMIN

        class Ok(api.Response):
            schema = api.Map(api.Array(api.String, "Banned network"), "Array of banned networks")


class ServiceProxyBannedResources(Api.Path("/service/banned_resources")):
    class Get(api.Get):
        """ List of resources banned in proxy """
        __security__ = ctu.Restriction.ADMIN

        class Ok(api.Response):
            schema = api.Map(api.Array(api.String, "Banned resource type"), "Array of banned resource types")


class ServiceProxyWhitelistResources(Api.Path("/service/whitelist_resources")):
    class Get(api.Get):
        """ List of resources allowed in proxy """
        __security__ = ctu.Restriction.ADMIN

        class Ok(api.Response):
            schema = api.Map(api.Array(api.String, "Allowed resource type"), "Array of allowed resource types")


class ServiceTasksEnqueuedCpuPreferences(Api.Path("/service/tasks/enqueued/cpu_preferences")):
    class Get(api.Get):
        """ Tasks enqueued cpu preferences """

        class Ok(api.Response):
            schema = api.Map(api.Integer, "Tasks enqueued cpu preferences")


class ServiceTasksEnqueuedQueueTime(Api.Path("/service/tasks/enqueued/queue_time")):
    class Get(api.Get):
        """ Tasks enqueued queue time """

        class Ok(api.Response):
            schemas = schemas.service.TasksEnqueuedQueueTime


class ServiceTelegraBotOwnerLastVisit(Api.Path("/service/telegram/owner_last_visit")):
    class Get(api.Get):
        """ Telegram owner last visit time """
        __security__ = ctu.Restriction.ADMIN

        class Ok(api.Response):
            __content__ = rest.Client.PLAINTEXT
            schemas = api.DateTime("Telegram owner last visit time")


class ServiceManageApiQuota(Api.Path("/service/quota/api/{user_name}")):
    class Get(api.Get):
        """Get api quota"""
        user_name = api.String("Name of user", required=True, scope=api.Scope.PATH)

        class Ok(api.Response):
            data = schemas.service.QuotaApi("Request data", required=True)

    class Put(api.Put):
        """Change api quota"""
        __security__ = ctu.Restriction.ADMIN

        user_name = api.String("Name of user", required=True, scope=api.Scope.PATH)
        data = schemas.service.QuotaApi("Request data", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            pass
