""" Operations on task objects """

from __future__ import absolute_import

from sandbox.common import api
import sandbox.common.types.user as ctu

from .api import Api
from .common import FilterBaseMultiOrder, Ids
from . import schemas


class TaskList(Api.Path("/task")):

    class Get(FilterBaseMultiOrder):
        """ Get task list """

        type = api.Array(api.String, "Task type(s)")
        id = Ids("Task Id(s)")
        status = api.Array(api.String, "Comma-separated task statuses")
        parent = Ids("Parent Id(s)")
        scheduler = api.Array(api.Integer, "Scheduler Id(s)")
        template_alias = api.String("Template alias")
        requires = Ids("Resource Id(s)")
        host = api.String("Host Id")
        arch = api.String("Client architecture")
        author = api.String("Task author")
        owner = api.String("Task owner")
        priority = api.String("Colon-separated class and subclass of task priority")
        desc_re = api.String("Regular expression for task description")
        model = api.String("CPU model")
        children = api.Boolean("Include child tasks flag")
        hidden = api.Boolean("Include hidden tasks flag")
        se_tag = api.String("Simultaneous execution tag")
        important = api.Boolean("Include only important tasks flag")
        created = api.DateTimeRange("Task creation time range: pair of ISO 8601 dates separated with `..`")
        updated = api.DateTimeRange("Task update time range: pair of ISO 8601 dates separated with `..`")
        input_parameters = api.String("Input parameters in format of dictionary: '{\"<name>\": <value>, ...}'")
        output_parameters = api.String("Output parameters in format of dictionary: '{\"<name>\": <value>, ...}'")
        any_params = api.Boolean("Accept task if any parameter matches, all parameters otherwise")
        tags = api.Array(api.String, "Task tags")
        all_tags = api.Boolean("Accept tasks with if it has all the tags, any of the tags otherwise")
        hints = api.Array(api.String, "Task hints")
        all_hints = api.Boolean("Accept tasks with if it has all the hints, any of the hints otherwise")
        fields = api.Array(api.String, "Choose only these fields")
        release = api.Array(api.String, "Release type(s). In case of provided, filter by status will be ignored")
        semaphore_acquirers = api.Id("Id of semaphore acquired by tasks", required=False)
        semaphore_waiters = api.Id("Id of semaphore waited by tasks", required=False)

        class Ok(api.Response):
            schema = schemas.task.TaskList
            x_matched_records = api.Header(api.Integer("Number of objects matched by the filter"))

    class Post(api.Post):
        """ Create task """

        data = schemas.task.TaskNew("New task data", required=True, scope=api.Scope.BODY)

        class Created(api.Response):
            schema = schemas.task.Task
            location = api.Header("URL of newly created task")

        class MovedPermanently(api.Response):
            location = api.Header("URL of duplicated task")

        class BadRequest(api.Response):
            """ Bad parameters """

        class Forbidden(api.Response):
            """ Requester cannot create task in on behalf of specified user """


class TaskAuditList(Api.Path("/task/audit")):

    class Get(api.Get):
        """ Get task audit records """

        id = Ids("Comma-separated task Ids", default=[])
        since = api.DateTime("Start of required time interval, UTC, ISO 8601")
        to = api.DateTime("Finish of required time interval, UTC, ISO 8601")
        remote_ip = api.String("Filter audit items by source host ip address")

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskAuditItem)


class TaskCurrent(Api.Path("/task/current")):

    class Get(api.Get):
        """ Current (bound to session) task info

        Available for requests with session OAuth token only.
        """

        __security__ = ctu.Restriction.TASK

        class Ok(api.Response):
            schema = schemas.task.Task

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskCurrentAudit(Api.Path("/task/current/audit")):

    class Post(api.Post):
        """ Create new audit record for current task """

        __security__ = ctu.Restriction.TASK

        body = schemas.task.TaskAuditNew("Audit object", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Status switched successfully """

        class Forbidden(api.Response):
            """ Allowed only with task session token """

        class BadRequest(api.Response):
            """ Incorrect status """

        class NotFound(api.Response):
            """ Task not found """

        class Conflict(api.Response):
            """ Conflict on status change """


class TaskCurrentContext(Api.Path("/task/current/context")):

    class Put(api.Post):
        """ Update context for current task """

        __security__ = ctu.Restriction.TASK

        body = api.Object("Task context", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Task context successfully updated """

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskCurrentContextValue(Api.Path("/task/current/context/value")):

    class Put(api.Post):
        """ Update one value for given key for current task context """

        __security__ = ctu.Restriction.TASK

        body = schemas.task.CurrentContextValue("Task context field", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Context value of the current task successfully updated """

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskCurrentExecution(Api.Path("/task/current/execution")):

    class Put(api.Post):
        """ Update current task execution info """

        __security__ = ctu.Restriction.TASK

        body = schemas.task.TaskExecutionUpdate(required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Execution info of the current task successfully updated """

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class Task(Api.Path("/task/{id}")):

    class Get(api.Get):
        """ Task detailed info """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.task.Task

        class NotFound(api.Response):
            """ Task not found """

    class Put(api.Put):
        """ Task info update """

        id = api.Id("Task Id", scope=api.Scope.PATH)
        data = schemas.task.TaskUpdate(scope=api.Scope.BODY)

        class Ok(api.Response):
            """ If X-Request-Updated-Data is set in request then updated task info wil be returned in response """

            schema = schemas.task.Task

        class NoContent(api.Response):
            """ Task successfully updated """

        class BadRequest(api.Response):
            """ Incorrect task Id """

        class NotFound(api.Response):
            """ Task not found """


class TaskResources(Api.Path("/task/{id}/resources")):

    class Get(api.Get):
        """ Task resources info

        Redirects to task resources page with correspondent filters
        """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class MovedPermanently(api.Response):
            location = api.Header("Url to get resources data")

        class NotFound(api.Response):
            """ Task not found """


class TaskRelease(Api.Path("/task/{id}/release")):

    class Get(api.Get):
        """ Task release info and template """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.task.TaskRelease()

        class NotFound(api.Response):
            """ Task not found """


class TaskCurrentRelease(Api.Path("/task/current/release")):
    class Post(api.Post):
        """ Set release field to task and add released attributes to resources """

        body = schemas.task.TaskReleaseCreate(scope=api.Scope.BODY)

        class Created(api.Response):
            """ Release fields created """

    class Delete(api.Delete):
        """ Delete release field from task and remove released attributes from resources """

        class NoContent(api.Response):
            """ Release info deleted """


class TaskAudit(Api.Path("/task/{id}/audit")):

    class Get(api.Get):
        """ Task audit records info """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskAuditItem)

        class NotFound(api.Response):
            """ Task not found """

    class Post(api.Post):
        """ Create new audit record for task """

        __security__ = ctu.Restriction.AUTHENTICATED

        id = api.Id("Task Id", scope=api.Scope.PATH)
        body = schemas.task.TaskAuditNew(required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            """ Audit record successfully created """


class TaskAuditHosts(Api.Path("/task/{id}/audit/hosts")):

    class Get(api.Get):
        """ Hosts where task was executed """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskAuditHostsItem)

        class NotFound(api.Response):
            """ Task not found """


class TaskCustomFields(Api.Path("/task/{id}/custom/fields")):

    class Get(api.Get):
        """ Task custom fields data """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskCustomFieldMeta)

        class NotFound(api.Response):
            """ Task not found """

    class Post(api.Post):
        """ Validate task custom fields data """

        id = api.Id("Task Id", scope=api.Scope.PATH)
        body = api.Array(schemas.task.TaskFieldValidateItem, "qwer", required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskFieldValidateResultItem)

        class NotFound(api.Response):
            """ Task not found """

    class Put(api.Put):
        """ Update task custom fields """

        __security__ = ctu.Restriction.AUTHENTICATED

        id = api.Id("Task Id", scope=api.Scope.PATH)
        body = api.Array(schemas.task.TaskFieldValidateItem, required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskFieldValidateResultItem)

        class NotFound(api.Response):
            """ Task not found """


class TaskTags(Api.Path("/task/{id}/tags")):
    class Post(api.Post):
        """ Add tags to task """
        id = api.Id("Task Id", scope=api.Scope.PATH)
        body = api.Array(api.String, required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Tags successfully added """

        class NotFound(api.Response):
            """ Task not found """

    class Delete(api.Post):
        """ Remove tags from task """
        id = api.Id("Task Id", scope=api.Scope.PATH)
        body = api.Array(api.String, required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Tags successfully deleted """

        class NotFound(api.Response):
            """ Task not found """


class TaskTagsTag(Api.Path("/task/{id}/tags/{tag}")):
    class Delete(api.Delete):
        """ Delete tag from task """
        id = api.Id("Task Id", scope=api.Scope.PATH)
        tag = api.String("Task tag", required=True, scope=api.Scope.PATH)

        class NoContent(api.Response):
            """ Tag successfully deleted """

        class NotFound(api.Response):
            """ Task not found """


class TaskHints(Api.Path("/task/{id}/hints")):
    class Post(api.Post):
        """ Add hints to task """
        id = api.Id("Task Id", scope=api.Scope.PATH)
        body = api.Array(api.String, required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Hints successfully added """

        class NotFound(api.Response):
            """ Task not found """


class TaskCurrentHints(Api.Path("/task/current/hints")):
    class Post(api.Post):
        """ Add hints to the current task """

        __security__ = ctu.Restriction.TASK

        body = api.Array(api.String, required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Hints successfully added """

        class NotFound(api.Response):
            """ Task not found """


class TaskCustomFooter(Api.Path("/task/{id}/custom/footer")):

    class Get(api.Get):
        """ Task custom footer data """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskReport)

        class NotFound(api.Response):
            """ Task not found """


class TaskReport(Api.Path("/task/{id}/reports/{label}")):

    class Get(api.Get):
        """ Task report data """

        id = api.Id("Task identifier", scope=api.Scope.PATH)
        label = api.String("Report label", scope=api.Scope.PATH, required=True)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskReport)

        class NotFound(api.Response):
            """ Report with this label does not exist """


class TaskOutput(Api.Path("/task/{id}/output")):

    class Get(api.Get):
        """ Task output fields data """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskCustomFieldMeta)

        class NotFound(api.Response):
            """ Task not found """


class TaskContext(Api.Path("/task/{id}/context")):

    class Get(api.Get):
        """ Task context """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = api.Object

        class NotFound(api.Response):
            """ Task not found """


class TaskQueue(Api.Path("/task/{id}/queue")):

    class Get(api.Get):
        """ Task in queue position info """

        id = api.Id("Task Id", scope=api.Scope.PATH)
        all = api.Boolean("Return info about clients not suitable for the task")

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskQueueItem)

        class NotFound(api.Response):
            """ Task not found """


class TaskDependant(Api.Path("/task/{id}/dependant")):

    class Get(api.Get):
        """ Task dependencies list

        Redirects to correspondent resource list page
        """

        id = api.Id("Task Id", scope=api.Scope.PATH)
        limit = api.Integer("Maximum number of returned entries", default=3000)
        offset = api.Integer("Skip this number of entries", default=0)

        class MovedPermanently(api.Response):
            location = api.Header("Url to get task dependencies")

        class NotFound(api.Response):
            """ Task not found """


class TaskChildren(Api.Path("/task/{id}/children")):

    class Get(api.Get):
        """ Child task list

        Redirects to correspondent task list page
        """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class MovedPermanently(api.Response):
            location = api.Header("Url to get child task list")

        class NotFound(api.Response):
            """ Task not found """


class TaskTriggerTime(Api.Path("/task/{id}/trigger/time")):

    class Get(api.Get):
        """ Get effective trigger on time elapsed """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.task.TaskTimeTrigger

        class NotFound(api.Response):
            """ Task not found """


class TaskCurrentTriggerTime(Api.Path("/task/current/trigger/time")):

    class Post(api.Post):
        """ Create trigger on time elapsed """

        __security__ = ctu.Restriction.TASK

        body = schemas.task.TaskTimeTriggerCreate("Trigger parameters", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Trigger successfully created """

        class BadRequest(api.Response):
            """ Bad arguments """

        class Conflict(api.Response):
            """ Trigger already exists """

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskTriggerTask(Api.Path("/task/{id}/trigger/task")):

    class Get(api.Get):
        """ Get effective trigger on tasks' status changes """

        id = api.Id("Task Id", scope=api.Scope.PATH)

        class Ok(api.Response):
            schema = schemas.task.TaskTaskTrigger

        class NotFound(api.Response):
            """ Task not found """


class TaskCurrentTriggerTask(Api.Path("/task/current/trigger/task")):

    class Post(api.Post):
        """ Create trigger on tasks' status changes """

        __security__ = ctu.Restriction.TASK

        body = schemas.task.TaskTaskTriggerCreate("Trigger parameters", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Trigger successfully created """

        class BadRequest(api.Response):
            """ Bad arguments """

        class NotAcceptable(api.Response):
            """ Nothing to wait """

        class Conflict(api.Response):
            """ Trigger already exists """

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskCurrentTriggerOutput(Api.Path("/task/current/trigger/output")):

    class Post(api.Post):
        """ Create trigger on changes of tasks' output fields """

        __security__ = ctu.Restriction.TASK

        body = schemas.task.TaskOutputTriggerCreate("Trigger parameters", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Trigger successfully created """

        class BadRequest(api.Response):
            """ Bad arguments """

        class NotAcceptable(api.Response):
            """ Nothing to wait """

        class Conflict(api.Response):
            """ Trigger already exists """

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskCurrentOutput(Api.Path("/task/current/output")):

    class Get(api.Get):
        """ Current task output fields data """

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskCustomFieldMeta)

    class Put(api.Put):
        """ Update current task output fields """

        __security__ = ctu.Restriction.TASK

        body = api.Array(schemas.task.TaskOutputFieldValidateItem, required=True, scope=api.Scope.BODY)

        class Ok(api.Response):
            schema = api.Array(schemas.task.TaskFieldValidateResultItem)

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskProlongate(Api.Path("/task/{id}/prolongate")):
    class Post(api.Post):
        """ Prolong task kill timeout """

        __security__ = ctu.Restriction.AUTHENTICATED

        id = api.Id("Task Id", scope=api.Scope.PATH)
        body = api.Integer("Number of seconds to prolong by", required=True, scope=api.Scope.BODY)

        class NoContent(api.Response):
            """ Task successfully prolongated """

        class BadRequest(api.Response):
            """ Bad arguments """


class TaskCommit(Api.Path("/task/current/commit")):
    class Post(api.Post):
        """ Commit task update (internal interface) """

        __security__ = ctu.Restriction.TASK

        class NoContent(api.Response):
            """ Task has been updated """

        class Forbidden(api.Response):
            """ Allowed only with task session token """


class TaskSemaphores(Api.Path("/task/current/semaphores")):
    class Delete(api.Delete):
        """ Release currently acquired semaphores """

        __security__ = ctu.Restriction.TASK

        class NoContent(api.Response):
            """Request has been successfully processed"""

        class Forbidden(api.Response):
            """ Allowed only with task session token """
